################################################################
#
#           Code for various parsing applications including
#           code for parsing one term or formula 
#
################################################################
#
# Each parse begins with a header.  Each header begins with a 
# tag which is one of the following numbers.
#
# Tags which appear only at the top level of a complete parse:
#
#  1. Term Definor                
#  2. Formula Definor             
#  3. Connector                   
#  4. Right Brace                 
#  5. Unrecognized constant       
#  6. Right Parenthesis           
#  7. Other known constant        
#  8. Term Seeking Notarian       
#  9. Formula Seeking Notarian    
# 10. Variable          (Object variable for terms)
# 11. Sentence variable (Boolean variable for formulas)
# 12. Function symbol, n-ary schemator n > 0 
# 13. Predicate symbol, n-ary schemator n > 0  
# 14. Noun             (Length 1 constant term) 
# 15. Boolean Constant (Length 1 constant formula)
# 16. Left Parenthesis        
# 17. Introductor      
# 18. Decimal Numeration
# 19. Left Scope Bracket 
# 20. Right Scope Bracket   
# 21. Colon 
# 22. Semi-colon
# 23. Ignore Token  (Not parsed)
# 24. Left Brace
# 25. Eqalign Token (Discard left brace)
# 26. LaTeX Font modifier
# 27. Sqrt Token (Save left brace) 
# 28. Revised notarian (for terms)
# 29. Revised notarian (for formulas)
# 30. Flanker
#
# Tags which may appear at any level of a complete parse:
#
#  40. Term       (Not length 1) 
#  41. Formula    (Not length 1) 
#  42. Schematic Term        
#  43. Schematic Formula    (Not zero-ary) 
#  44. Parade Term
#  45. Parade Formula
#  47. Newly  Defined Form
#  48. Scope 
#  49. Undefined Expression (2nd level only)  
#  50. Undefined Parade (3rd level only)  
#  51. New  Definition
#  61. Ignored Expression
# 
# Tags which may appear only at the beginning of an incomplete parse:
#
# -1. Non-Parenthetical Expression -11. New Definition      
# -2. Parenthetical Expression     -12. Undefined Parade 
# -3. Undefined Expression         -13. 
# -4. Schematic Term               -14. Brace Expression
# -5. Schematic Formula           
# -6. Term Quantifying Form       
# -7. Formula Quantifying Form    
# -8. Scope                        -18. New Notarian Term 
# -9. Parade                       -19. New Notarian Formula 
#-10. EqAlign Expression           -20. 
#
#
################################################################
#
import pattern,re,copy
#
#
#  This database is loaded from the dfs file.  It is modified by parse
#  but not by check.  It is initialized by resetdefs and resetprops.  

mathdb = [] 
mathtf = {}

MD_SYMTYPE = 0    #  Dictionary storing the type of each symbol
MD_PRECED = 1     #  Dictionary storing the precedence of each connector
MD_DEFS = 2       #  Dictionary storing a list of definienda for each introductor
MD_ARITY = 3      #  Dictionary storing the arity of each schemator 
MD_IDOPS = 4      #  List of Identity operators
MD_TRMUL = 5      #  Dictionary mapping 2-tuples of transitive ops to a transitive op
MD_REDFILE = 6    #  List of reduction rule files  
MD_TRFILE = 7     #  List transitive rule locations 
MD_REFD = 8       #  Dictionary of external file reference definitions 
MD_MACR = 9       #  Dictionary of user macro definitions
MD_PROPGET = 10   #  Properties acquisition schedule 
MD_TRPRS = 11     #  List of Transitive precedences 
MD_RFILE = 12     #  List of inference rule files
MD_NRYPROP = 13   #  Unused
MD_DLMATCH = 14   #  Dictionary storing left match for each right delimiter
MD_CTDEFS = 15    #  Dictionary storing a list of definienda for connectors
MD_CTCHK = 16     #  Dictionary of connector parsers by precedence
MD_CTHNDL = 17    #  Dictionary of connector expanders by precedence
MD_PSPEECH = 18   #  Dictionary defining parts of speech
MD_LEN = 19       #  Length of mathdb

# Of these database entries, many are local to the associated file, but some
# are inerited and merge with other database entries.  The following are
# local only: MD_IDOPS, MD_TRMUL, MD_CAOPS, MD_TRFILE, MD_REFD, MD_TRPRS, MD_RFILE.
#
# The following fast moving variable keeps track of which numbers have been
# used already as variable tags.   
#
newvarnum = 0

#The following are tokens added so that TeX spacing characters can be used.
#ignore_tokens = ['\,','\>','\;','\!']

reference_punctuator_list = [',',';','\\C','G','H','!','D','P','A','S','T','U',')','(','),',',(',';(','+','-',':','|']

# LaTeX font modifiers may accumulate.  This variable records which
# modifiers are currently in effect.

bracestack = []
braceprefix = ''
current_fontmods = frozenset([])
current_fontnum = 0
fontmoddict = {}
needed_syntax_handlers = set()

#Just a minimal test set for now:
allowed_fontmodifiers = ['\\cal', '\\mathbf']



def makemathdb():

	td = {}
	precedence = {}
	defs = {}
	arity = {}

#######################################################
#
#  Initialize Properties
#
#######################################################

	td['dft'] = 1  #These are not real tokens. They would be
	td['dff'] = 2  #broken apart if they occurred in a document.
	precedence['dff'] = 0
	precedence['dft'] = 0
	td['xvar'] = 10
	td['pvar'] = 11
	td['('] = 16
	td[')'] = 6
	td['{'] = 24
	td['}'] = 4
	td['\\ls'] = 19
	td['\\rs'] = 20
	td[':']= 21 
	td[';']= 22 
	td['\\}']= 7 
	td['\\ten'] = 14
#Brace prefixing tokens
	td['\\eqalign'] = 25 #discard left brace
	td['\\noparse'] = 25 #discard left brace
	td['\\sqrt'] = 27  #save left brace
#ignore_tokens 
	td['\\,'] = 23
	td['\\>'] = 23
	td['\\;'] = 23
	td['\\!'] = 23
	td['\\cr']= 23
	td['&']   = 23
#Need display formulas to tolerate the period:
	td['.'] = 7

	identity_ops = []

	trans_mult = {}

	reduction_rules_files = []

	transitive_rules_files = []

	reference_dictionary = {'0' : 'common.tex'}

	user_dictionary = {}

	prop_acquisitions = {}

	transitive_precedences = [] 

	rules_file_list = []

	nary_propflag = {}

#	nary_typepair = {}
	delim_match= {}

	delim_match[')'] = '('

	delim_match['}'] = '{'

	connector_defs = {}

	connector_parsers = {}

	connector_expanders = {}

	speech_part = {}

	nexus_precedences = []

	nexus_checkers = []

	return [td, precedence, defs, arity,identity_ops,trans_mult,
			reduction_rules_files,transitive_rules_files, reference_dictionary, 
			user_dictionary, prop_acquisitions, transitive_precedences, rules_file_list,
			nary_propflag, delim_match, connector_defs, connector_parsers,
			connector_expanders, speech_part]


def dbmerge(mathdb,db):
	if len(db)!= MD_LEN:
		print("Warning: format does not match.")
#Dictionaries to update
	for entry in [MD_SYMTYPE, MD_ARITY, MD_PRECED, MD_CTCHK, MD_PSPEECH]:
		mathdb[entry].update(db[entry])
#Dictionaries with mergible entries 
	for entry in [MD_DEFS, MD_CTDEFS, MD_CTCHK, MD_CTHNDL]: 
		for k in db[entry]:
			if k in mathdb[entry]:
				for y in db[entry][k]:
					if y not in mathdb[entry][k]:
						mathdb[entry][k].append(y)
			else:
				mathdb[entry][k] = db[entry][k]

def symtype(token):
	assert type(token) is str
	td = mathdb[MD_SYMTYPE]
	retval = td.get(token, 0)
	if retval:
		return retval
	if pattern.TeX_leftdelimiter.match(token):
		td[token] = 16
		return 16
	if pattern.TeX_rightdelimiter.match(token):
		td[token] = 6
		matching_token = pattern.rmatch(token)
		td[matching_token] = 16
		mathdb[MD_DLMATCH][token] = matching_token
		return 6
	b = validschemator(token)
	if b:
		arity = mathdb[MD_ARITY]
		td[token] = b[0]
		arity[token] = b[1]
		return  b[0]
	if validvar(token):
		td[token] = 10
		return 10
	if validnum(token) and len(token) > 1:
		td[token] = 18
		return 18
	if validnum(token):
		td[token] = 14
		return 14
	if token in allowed_fontmodifiers:
		td[token] = 26 
		return 26
	td[token] = 5
	return 5 
	
def validschemator(token):
	schemm = pattern.newschem.match(token)
	if schemm:
		arity = int(schemm.group(1))
		if token[1] == 'w':
			return (12, arity)
		elif arity == 0:
			return (11, arity)
		else:
			return (13, arity)
	genschemm = pattern.genschem.match(token)
	if genschemm:
		arity = len(genschemm.group(2)) + 1
		sym2 = genschemm.group(1)[0]
		if sym2 in ['p','q','r']:
			return ( 13, arity)
		else:
			return ( 12, arity)
	gensentm = pattern.gensent.match(token)
	if gensentm:
		return (11, 0)
	if len(token) < 5:
		return 0
	if token[0] != '\\':
		return 0
	if token[1] not in ['o','p','q','r','s','t','u','v','w']:
		return 0
	if len(token) == 6 and token[5] != 'p':
		if token[3:6] != 'var':
			return 0
		if token[1] not in ['p','q','r','s','t']:
			return 0
		if token[2] not in ['p','q','r','s','t']:
			return 0
		return (11,0)
	if token[3:5] != 'ar' :
		return 0
	if token[2] == 'v':
		if token[1] in ['o','p','q','r','s','t']and len(token) == 5:
			return (11,0)
		else:
			return 0
	elif token[2] != 'b':
		return 0
	if token[1] not in ['p','q','r','u','v','w']:
		return 0
	tail = token[5:]
	lt = len(tail)
	if tail.count('p') == lt:
		if token[1] in ['p','q','r']:
			return(13, 1+lt)
		else:
			return (12, 1+lt)
	else:
		return 0

def validvar(token):
	if token[0].isalpha() :
		return True
	elif  token in allowed_variables:
		return True
	tokenm = pattern.token.match(token)
	if tokenm.group(3) in allowed_variables:
		return True
	return False
#
# This is the default list only.  The directive %variable: may be
# used to declare additional variables
#
allowed_variables = ['\\alpha','\\beta','\\gamma','\\delta','\\epsilon',
'\\varepsilon','\\zeta','\\eta','\\theta','\\vartheta','\\iota',
'\\kappa','\\lambda','\\mu','\\nu','\\xi','\\pi','\\varpi',
'\\rho', '\\varrho','\\sigma', '\\varsigma','\\tau','\\upsilon',
'\\phi', '\\varphi','\\chi','\\psi','\\Gamma','\\Delta','\\Theta',
'\\Lambda', '\\Xi','\\Pi','\\Sigma','\\Upsilon','\\Phi', '\\Psi',
'\\imath','\\jmath','\\ell']

def validnum(token):
	return token.isdigit()

def varslot(deflist, slot_number):
	for definiendum in deflist:
		x = definiendum[slot_number]
		if type(x) is list or symtype(x) in [10,11]:
#		if definiendum[slot_number] is a variable:
			return True 
	return False 

def nodify(token, tree):
	global braceprefix, bracestack
#
#	print("token = ", token)
	precedence = mathdb[MD_PRECED]
	defs = mathdb[MD_DEFS]
	arity = mathdb[MD_ARITY]
	n = symtype(token)
	if n == 18:
		x = decimalnode(token)
	elif n == 12:
		x = [[-4, arity[token]], [[n], token ]]
	elif n == 13:
		x = [[-5, arity[token]], [[n], token ]]
	elif n == 30:
#		print("tree ==========", tree)
#		print("tree[-1] ======", tree[-1])
#		print("token =========", token)
		if len(tree) > 1 :
			if len(tree[-2]) < 2:
				x = [[-2 , []], [[n], token ]]  #Left flank
#				assert tree[-2][0][0] == -8
			elif tree[-2][1][1] == token and tree[-1][0][0] == -9:
#				print("tree[-2][1][1] ======", tree[-2][1][1])
				if len(tree[-1]) == 1:
					x = [[-2 , []], [[n], token ]]  #Left flank
				elif tree[-1][-1][0][0] == 3:
					x = [[-2 , []], [[n], token ]]  #Left flank
				else:
					x = [[n], token ]               #Right flank
			else:
				x = [[-2 , []], [[n], token ]]  #Left flank
		else:
			x = [[-2 , []], [[n], token ]]  #Left flank

	elif n == 16:
		if tree == []:
			x = [[-2 , []], [[n], token ]]
		elif tree[-1][0][0] == -1:
			if varslot(tree[-1][0][1], len(tree[-1])):
				x = [[-2 , []], [[n], token ]]
			else:
				x = [[n], token ]
		elif tree[-1][0][0] == -3:
			x = [[n], token ]
		else:
			x = [[-2 , []], [[n], token ]]
	elif n == 24 :                      #Left Brace
		assert braceprefix in ['','\eqalign','\sqrt']
		if braceprefix and symtype(braceprefix) == 25:  #Brace following eqalign 
			bracestack.append([])
			x = [[-14, []], [[n], token ]]
			assert False
		elif braceprefix and symtype(braceprefix) == 27:   #Brace following sqrt 
			x = [[-14 , []], [[n], token ], [[27], braceprefix]]
			braceprefix = ''
		elif bracestack and bracestack[-1] == '\\cal':
			bracestack.append([])
			assert False
		else:                       #Left brace for a new form or for a font scope
			x = [[-14 , []], [[n], token ]]
#			print("braceprefix ====", braceprefix)
#			print("junk bracestack ===", bracestack)
#			bracestack.append([])
	elif n == 25 :
			assert False
			x = [[-10 , []], [[n], token ]]
	elif n == 17 :
		if tree == []:
			x = [[-1, defs[token]], [[n], token ]]
		elif tree[-1][0][0] == -1:
			if varslot(tree[-1][0][1], len(tree[-1])):
				x = [[-1, defs[token]], [[n], token ]]
			else:
				x = [[n], token ]
		elif tree[-1][0][0] == -3:
			x = [[n], token ]
		else:
			x = [[-1, defs[token]], [[n], token ]]
	elif n == 28 :
		x = [[-18, defs[token]], [[n], token]]
	elif n == 8:
		x = [[-6, defs[token]], [[n], token ]]
	elif n == 9:
		x = [[-7, defs[token]], [[n], token ]]
	elif n in [1,2,3] :
		x = [[n], token, precedence[token]]
	elif n == 10 and current_fontmods:
		modified_token = token + "." + str(current_fontnum)
		mathdb[MD_SYMTYPE][modified_token] = 10
		x = [[n], modified_token]
	else:
		# Even unknown constants are passed on as complete 
		x = [[n], token]
	return x

def notarian_match(dfn, partialmatch, nextitem):
#print("Notarian_Match")
#	print(dfn)
#	print("    vs:", partialmatch)
#	input("here")
	semicolon_spot = dfn.index(';')
	if nextitem[0][0] == -8:
		if semicolon_spot != len(partialmatch) + 1:
			return False
		partialmatch = partialmatch + [nextitem[1]]
	elif nextitem[0][0] > 0:
		partialmatch = partialmatch + [nextitem]
#	if len(partialmatch) > semicolon_spot:
#		return False
	for k in range(1, len(partialmatch)):
		if type(dfn[k]) is list:
			if dfn[k][0][0] == 42:
				if type(partialmatch[k]) is list:
					if partialmatch[k][0][0] not in [40,42,10,14]:
						return False
				elif type(partialmatch[k]) is str:
					if symtype(partialmatch[k]) not in [10, 14]:
						return False
			elif dfn[k][0][0] == 43:
				if type(partialmatch[k]) is list:
					if partialmatch[k][0][0] not in [41,43,11,15]:
						return False
				elif type(partialmatch[k]) is str:
					if symtype(partialmatch[k]) not in [11, 15]:
						return False
		elif symtype(dfn[k]) == 10:
				if type(partialmatch[k]) is list:
					if partialmatch[k][0][0] not in [40,42,10,14]:
						return False
				elif type(partialmatch[k]) is str:
					if symtype(partialmatch[k]) not in [10, 14]:
						return False
		elif symtype(dfn[k]) == 11:
				if type(partialmatch[k]) is list:
					if partialmatch[k][0][0] not in [41,43,11,15]:
						return False
				elif type(partialmatch[k]) is str:
					if symtype(partialmatch[k]) not in [11, 15]:
						return False
		elif dfn[k] != partialmatch[k][1]:
			return False
#	print("TRUE")
	return True

def decimalnode(token):
	precedence = mathdb[MD_PRECED]
	n = len(token)
	if n == 1:
		return [[14,[]], token]
	retval = token[0]
	for k in range(1,n):
		retval = [[44,precedence['+']], 
                [[44, precedence['\\cdot']], retval, '\\cdot', '\\ten'], '+', token[k]]
	return [[40, []], '(' , retval, ')' ]

def addtoken(tree, token):
#	print("addtoken ===", token)
#	print("tree ===", tree)
	global bracestack, braceprefix, current_fontmods, current_fontnum
	userdict = mathdb[MD_MACR]
	pfchk_token = userdict.get(token, token)
	if '\\noparse' in bracestack:
		if symtype(pfchk_token) == 24: #Left Brace 
			bracestack.append([])
		elif symtype(pfchk_token) == 4: #Right Brace
			bracestack.pop()
		return 1
	if symtype(pfchk_token) == 24:  #Left Brace 
		if braceprefix == '\\eqalign': 
			braceprefix = '' 
			bracestack.append([])
			return 1
		elif braceprefix == '\\noparse':
			bracestack.append(braceprefix)
			braceprefix = '' 
			return 1
		elif braceprefix == '\\sqrt': 
			bracestack.append(braceprefix)
#			braceprefix = '' 
		elif braceprefix:
			assert False, "What is " + braceprefix + "?"
			pass
		else:                        # \sqrt prefix or no prefix
#			bracestack.append(len(bracestack))
			bracestack.append('+')
			assert braceprefix == ''
			braceprefix = '' 
	elif braceprefix:  #Left Brace required
			print("Left brace expected after " + braceprefix)
			return 0
	elif symtype(pfchk_token) == 23:  #These tokens completely ignored
		return 1
	elif symtype(pfchk_token) in [25,27]: #Demands following left brace, like \eqalign or \sqrt
		braceprefix = pfchk_token
		return 1
	elif symtype(pfchk_token) == 26:  #Font modifier
		assert tree[-1][0][0] == -9, tree[-1][0][0]
		assert tree[-2][0][0] == -14, tree[-2][0][0]
		tree.pop()
		tree.pop()
		x = bracestack.pop()
		assert x == '+',"Brace stack pop yielded: " + x
		bracestack.append(pfchk_token)
		current_fontmods = current_fontmods | frozenset([pfchk_token])
		n = fontmoddict.get(current_fontmods, -1)
		if n == -1:
			n = len(fontmoddict) + 1
			fontmoddict[current_fontmods] = n
		current_fontnum = n
#				print("current_fontnum =========", current_fontnum)
#		promote(item, 61)
		return 1
	elif symtype(pfchk_token) == 4:  # Right Brace
		if not bracestack:
			print("No matching left brace")
			return 0
		if bracestack[-1] in allowed_fontmodifiers: # If there is a font modifier at this level
#			assert False
			fontmod = bracestack.pop()
			current_fontmods = current_fontmods - frozenset([fontmod])
			if current_fontmods:
				current_fontnum = fontmoddict[current_fontmods]
			else:
				current_fontnum = 0 
			return 1
		elif bracestack[-1] == []: #Don't send this into the parser
			bracestack.pop()
			return 1
		else:
			print("CCC")
			print(tree)
			pass # This is proofcheck math not LaTeX
	adn = addnode(tree, nodify(pfchk_token, tree))
#	print("Addnode returns =========", adn)
#	print("tree ===", tree)
	return  adn

def posneg(tree):
	neg_seen = False
	for x in tree:
		if x[0][0] < 0: neg_seen = True
		if neg_seen and x[0][0] > 0:
			return False
	return True

def addnode(tree, item):
#tree is a list which has one entry for each pending incomplete parse tree
#item is one complete parse
#
#	print("addnode: tree = ", tree)
#	print("addnode: item = ", item)
#	print("bracestack= ", bracestack)
	header = item[0]
	syntype = header[0]
#	print("addnode: syntype = ", syntype)
#	print()
#	okval = 1

# An unknown symbol which appears immediately following
# a single open parenthesis can only be the beginning
# of a definition which makes it an introductor.
# In this case it must be repackaged as an incomplete node.

	if syntype == 5:      # A new symbol appears
		print("New symbol:", item[1] )
		if len(tree) < 2:
			return 0
		if tree[0][0][0] != -2 or tree[1][0][0] != -9:
			return 0
		if len(tree) == 2: 
			if len(tree[1]) ==1:
				item = [[-3,[]], item] # Change new node to incomplete
				header = item[0]
				syntype = header[0]
#				print("New Introductor")
			else:
				return 0
		elif len(tree) == 3:
			if  tree[2][0][0]  in [-1, -3, -18, -19]: 
				pass  # Neither an introductor nor a connector. 
				# if tree[2][0][0] == -1: 
				# tree[2][0][0] = -3 will be done in nodecheck() 
			else:
				return 0
		elif len(tree) == 4 and len(tree[1]) == 1:
			# This must be a new connector.
			if tree[2][0][0] == -2 and  tree[3][0][0] == -12:
				pass
			elif tree[2][0][0] == -2 and  tree[3][0][0] == -9:
#				print "new connector"
				tree[3][0][0] = -12	
				item[0][0] = 3
				item.append(-2)
			else:
				return 0
		else:
			return 0

	if syntype == 30: #Flanker!
#		print("tree[-2] == ", tree[-2])
#		print("item =====", item[1])
#		print(tree[-2][1][1] == item[1])
		if tree != [] and tree[-1][0][0] == -9: 
			if tree[-1][-1][0][0] == 3:
				pass
			elif tree[-2][1][1] != item[1]: # check for right-left match
				pass
			else:
				crop = paradecrop(tree[-1], 0)
				if type(crop) is str:
#					if len(tree) == 4 and len(tree[3])== 2 and tree[3][1][0][0] == 50:
#						parsed_parade = tree.pop().pop()
#						current = tree.pop()
#						current.append(parsed_parade)
#						tree.append(current)
#					else:
#						print("tree ===", tree)
						print("Parade syntax error:", crop)
						return 0
				else:
					parsed_parade = tree.pop().pop()
					current = tree.pop()
					current.append(parsed_parade)
					tree.append(current)
#					print("tree3 ===", tree)

	if syntype == 6 or syntype == 4:# Recognize the end of a parade or a definition
		if len(tree) == 0:
			print("Error: Extra right paren")
			return 0
		if tree[-1][0][0]  ==-9: # Parade
			ld = mathdb[MD_DLMATCH]
			if ld[item[1]] != tree[-2][1][1]:
				print(str(item[1]) + " does not match with " + str(tree[-2][1][1]))
				return 0 
#			if not paradecrop(tree[-1], 0): #Calls paradecheck
#			for k in range(len(tree)):
#							print("tree[", k , "] = ", tree[k])
#			print("tree1 ===", tree)
			crop = paradecrop(tree[-1], 0)
#			print("crop ===========", crop)
#			print("tree2 ===", tree)
			if type(crop) is str:
				if syntype == 6 and len(tree) == 4 and len(tree[3])== 2 and tree[3][1][0][0] == 50:
					parsed_parade = tree.pop().pop()
					current = tree.pop()
					current.append(parsed_parade)
					tree.append(current)
				else:
#					print("tree ===", tree)
					print("Parade syntax error:", crop)
					return 0
			else:
				parsed_parade = tree.pop().pop()
				current = tree.pop()
				current.append(parsed_parade)
				tree.append(current)
#				print("tree3 ===", tree)
		elif tree[-1][0][0] == -12: #Undefined parade 
# 			print "Undefined parade", tree[-1]
			current = tree.pop()
			promote(current, 50)
			tree[-1].append(current)
		elif tree[-1][0][0] == -11: # New definition
			print(tree[-1])
			print("Should not reach this point")
			assert False
	
	if syntype in [1,2]: 
		if len(tree) < 2:
			return 0
		if len(tree) == 3:
			if tree[0][0][0]  == -2 and tree[1][0][0] == -9 and tree[2][0][0] == -3:
				# This is the only valid way out of a -3 state.
				# Change from a parade to a definition in the next if block 
				current = tree.pop()
				promote(current, 49)
				tree[-1].append(current)

	if syntype in [1,2]: # Change parades to definitions
		if tree[-1][0][0] == -9 and len(tree[-1]) == 2:
			if tree[-1][1][0][0] == 49:
				tree[-1][0][0] = -11

	if len(tree) > 1 and len(tree[-1]) > 1 and tree[-1][0][0] == -8: #End scope
#
# No need for recursion here since a scope 
# is not allowed at the end of a formula
#
#		print("tree = = = ", tree)
#		print("syntype = = = ", syntype)
#		input("here too?")
		lasthead = tree[-2][0]  #Header of the notarian expression
		if syntype == 22:       # Semi-colon
			if scopecheck(tree[-1]):
				if len(lasthead) == 2:# No length determination made yet
					lasthead.append(5)  # E xiA ; px qx
				elif lasthead[2] != 7:
					return 0
				current = tree.pop()
				promote(current, 48)
				tree[-1].append(current)   
			else:
				return 0
		elif syntype == 21:     # Colon
			if scopecheck(tree[-1]):
				if len(lasthead) > 2:
					return 0
				else:
					lasthead.append(4)  # E xiA : px
				current = tree.pop()
				promote(current, 48)
				tree[-1].append(current)   
			else:
				return 0
		elif syntype == 19:   # Left Scope Bracket
			if len(tree[-1]) == 2 and tree[-1][-1][0][0] in [10,14,40,42]:
				lasthead.append(7)  # E ux < xiA ; px >
				current = tree.pop()
				current = current.pop()  # Not a scope after all.
				tree[-1].append(current)   
			else:
				return 0
		elif syntype == 20:     #Right Scope Bracket
			if len(lasthead) <= 2 or lasthead[2] != 7:
				return 0
			elif scopecheck(tree[-1]):
				if len(tree[-2]) == 4 and tree[-2][-1][0][0]== 19:
					lasthead[2] = 6  # E ux < xiA >  , No px after all.
				else:
					return 0
				current = tree.pop()
				promote(current, 48)
				tree[-1].append(current)   
			else:
				return 0
		elif not (tree[-1][-1][0][0] in [3] or syntype in [3]):
			if len(lasthead) != 2:
				pass
			elif scopecheck(tree[-1]):
#				okval = 2
				lasthead.append(3)  # E xiA qx
				current = tree.pop()
				promote(current, 48)
				tree[-1].append(current)   
			else:
				return 0
		

################################################################
#
#    Main Algorithm 
#
################################################################
			
	if syntype in [-1,-2] and tree and len(tree[-1][0]) > 1:
	#
	# Check for a preceding internal occurrence of the introductor
	#
		deflist = tree[-1][0][1]
		if type(deflist) is list and tree[-1][0][0] in [-1,-3]:
			n = len(tree[-1])
#			print("deflist = ", deflist)
			for definiendum in deflist:
#				print("definiendum = ", definiendum)
#				print("n = ", n)
#				print("item = ", item)
#				print("tree[-1]=  ", tree[-1])
#				print("syntype == ", syntype)
				if len(definiendum) > n and definiendum[n] == item[1][1]: 
#						print("Correct")
						item = item[1]
						header = item[0]
						syntype = header[0]
						break
		
	if tree and tree[-1][0][0] in [-18, -19] and len(tree[-1][0])<3:
		df = tree[-1][0][1]  # Ordinarily this list has one element 
		                     # But not for notarians like, alm phi, alm # M phi, etc. 
#		print("tree[-1] ===========", tree[-1])
#		print("len(tree[-1]) ======", len(tree[-1]))
#		print("tree[-1][0] ===========", tree[-1][0])
#		print("len(tree[-1][0]) ======", len(tree[-1][0]))
#		print("item ===============", item)
#		print("syntype ============", syntype)
		partialmatch_list = [x for x in df if notarian_match(x,tree[-1], item)]
#		print("partialmatch_list ==", partialmatch_list)
#		input("Got here.")
		if len(partialmatch_list) > 1:
			print("More than one notarian matches")
#			return 1
		elif len(partialmatch_list) == 1:
			semicolon_spot = partialmatch_list[0].index(';')
#			print("df =================", df)
#			print("semicolon_spot =====", semicolon_spot )
			if len(tree[-1]) + 1 == semicolon_spot and syntype !=-8:
				tree[-1][0][1] = partialmatch_list
				tree.append([[-8,[]]])
#				print("tree ===========", tree)
#				input("Got here yes.")
		elif len(tree) == 3 and tree[0][0][0] == -2 and tree[1][0][0] == -9:
#			input("tree = = = " + str(tree))
			tree[-1][0] = [-3,[]]
		else:
			print("No notarian matches")
			return 0
#		if len(tree[-1]) == semicolon_spot - 1: #Check length of notarian form
#			tree.append([[-8,[]]])
#			input("Got here too.")

	if syntype == -14 and braceprefix == '\eqalign': #  Ignore!
#		bracestack.append([])
#		print("bracestack = = = ", bracestack)
		assert False
		return 1
	elif syntype < 0: # Incomplete nodes are just appended, not checked!
#	if syntype < 0: # Incomplete nodes are just appended, not checked!
		tree.append(item)
		if item[1][0][0] == 16 and len(item) == 2:
			tree.append([[-9,0]])
		if item[1][0][0] == 24 and len(item) == 2:
			tree.append([[-9,0]])
		if item[1][0][0] == 30 and len(item) == 2:
			tree.append([[-9,0]])
		if item[-1][0][0] in [19, 8, 9]:
			tree.append([[-8,[]]])
		if item[-1][0][0] in [28,29]:
			pass
#			tree.append([[-8,[]]])
		assert posneg(tree)
#		assert all([x[0][0] < 0 for x in tree])
#		for x in tree:
#			if x[0][0] > 0:
#				return 0
		return 1 
	elif syntype > 59:  # LaTeX stuff ignored by the parser
		return 1
	elif tree:  # When a complete node arrives it is added to the last incomplete
		# node, possibly completing it.  Recursion follows. 
		for i in range(len(tree)):
# Since known introductors can introduce unknown forms
# it catches errors sooner to check for unknown forms here.
			if tree[i][0][0] == -3:
				if i != 2:
					return 0
				if tree[0][0][0] != -2:
					return 0
		current = tree.pop()
		current.append(item)
		ndc = nodecheck(current)
		if ndc == 0:
#			print("tree[-1] =======", tree[-1])
#			print("current ========", current)
			return 0
		elif type(ndc) is str:
			print(ndc)
			return 0
		adn = addnode(tree,current)
		return adn
	else:       # If no incomplete nodes are left we are done.
		tree.append(item)
		return 1

def promote(node,newvalue): # Change incomplete node to complete
	for k in range(1, len(node)):
		if node[k][0][0] < 40:
			node[k] = node[k][1]
	node[0][0] = newvalue

def nodecheck(item):
# 
#item is a tree with a newly added node
# nodecheck determines whether it should
# converted to a complete node and does
# so if item is ready.  It returns 1 unless
# there is a parse error.
	global current_fontmods, current_fontnum, fontmoddict,braceprefix,bracestack
#
#	print("nodecheck:item ===============", item)
#	pprint(item)
	header = item[0]
	syntype = header[0]
	thisheader = item[-1][0]
	if syntype  == -1:
		r = deflistupdate(item)
		if r == 0 :
			header[0] = -3 # Undefined expression
			return 1
		if r == 1:
			d = item[0][1][0] 
			if len(d) == len(item) : 
				if d[0][0]  == 40:
					promote(item, 40)
				elif d[0][0]  == 41:
					promote(item, 41)
				else:
					raise SystemExit
#				item[0].remove(item[0][1])
				defs = mathdb[MD_DEFS]
				item[0][1][0] = defs[item[1]].index(item[0][1][0])
				for x in d[0][1][1:]:
					item[0][1].append(x)
#				item[0][1] = [defs[item[1]].index(item[0][1][0])]
		return 1
	elif syntype == -2: #Parenthetical expression
		if item[-1][0][0] == 45:
			return 1
		elif item[-1][0][0] == 44:
			return 1
		elif item[-1][0][0] == 47:
			return 1
		elif not(item[-1][0][0] == 6 or 
							 (item[-1][0][0] == item[1][0][0] == 30) and item[-1][1] == item[1][1]):
			return 0
#		print("item[-2][0][0] == ", item[-2][0][0])
#		print("item == " , item)
#		Checks already done.  Need get term vs. formula straight in next line.
		if item[1][0][0] == 30 and item[-2][0][0] in [10,14, 40, 42]:
			promote(item,40)
		elif item[-2][0][0] == 44:
			promote(item, 40)
		elif item[-2][0][0] == 45:
			promote(item, 41)
		elif item[-2][0][0] == 47:
			promote(item, 51)
		elif item[-2][0][0] == 50:
			promote(item, 49)
		else:
# This location is reached if x_ instead of x\_ is used"
			return 0
		return 1
	elif syntype == -3:  # Undefined Expression
		if not thisheader[0] in [1,2,6,14,15]:  
			return 1
	elif syntype in [ -4, -5]:  # Schematic Expressions 
		if thisheader[0] in [10, 14, 40, 42] : # A term 
			if len(item) == header[1] + 2: 
				promote(item, 38 - syntype) # Get values: 42 term, 43 formula 
			return 1
	elif syntype == -11:  # A New Definition
		lastheader = item[-2][0]
		if item[1][0][0] != 49:
			print("Mistake!")
			raise SystemExit
		if len(item) > 4:
			return 0
		elif len(item) == 3:
			return thisheader[0] in [1,2] 
		elif len(item) == 4:
#			print("item ====================", item)
			dcheck = definitioncheck(item)
			if type(dcheck) is str:
				return "Definition check failed: " + dcheck
			else:
				promote(item,47)
#				print("Definition check succeeded " + str(dcheck))
				return 1
	elif syntype == -12:  # Undefined Parade
		if thisheader[0] in [10,11,14,15,40,41,42,43]:
			if len(item) < 3:
				return 1
			if item[-2][0][0] in [10,11,14,15,40,41,42,43]:
				print("Error: Connector missing.")
				return 0
			else:
				return 1
		elif thisheader[0] in [3,5,7]:
			return 1 
	elif syntype == -10:  # Handle Eqalign 
#		print("item =======")
#		for x in item:
#				  print(x)
		assert False
		if len(item) == 2:
			assert thisheader[0] == '\\eqalign'
			pass
		elif len(item) == 3:
			if thisheader[0] ==  24: # Left brace
					return 1
			else:
					print("Left brace expected.")
					print("item =======" ,item)
					return 0
		elif thisheader[0] == 4:   # Right brace
			if len(item) > 4:
				del(item[0])
				for node in item[1:]:
					if type(node) is list and node[0][0] in [40,41]:
						y = node[:]
						break
				item[:] = y
				print("item =======" ,item)
				input("HALT")
				return 1
			else:
				return 0
		else:
			return 1
	elif syntype == -9:  # A Parade
		if thisheader[0] == 49:
			if len(item) == 2:
				item[0][0] = -11
				return 1
		elif thisheader[0] in [10,11,14,15,40,41,42,43]:
			if len(item) < 3:
				return 1
			if item[-2][0][0] in [10,11,14,15,40,41,42,43]:
				header[1] = 1000 # Precedence of the empty connector
				print(item)
				print("Warning: Connector missing.")
			return 1
		elif thisheader[0] == 7:
			return item[-1][1] + " used as infix. To do this set its precedence." 
		elif thisheader[0] in [5,44,45]:
			return 1 
		elif thisheader[0] in [1,2,3]:
			if len(item) == 3 and thisheader[0]in[1,2]and item[1][0][0] == 49:
				item[0][0] = -11
				return 1	
			if item[-1][2] >= header[1]: # compare with parade's max
				header[1] = item[-1][2] 
				return 1
			savelast = item.pop()
#			crop = paradecrop(tree[-1], 0)
			crop = paradecrop(item, savelast[2])
			if type(crop) is str: 
				print("Parade syntax error:",crop)
				return 0
			item.append(savelast)
			item[0][1] = savelast[2]
			return 1 
	elif syntype == -8:  # A scope 
		if thisheader[0] in [3,5,7,10,11,14,15,40,41,42,43]:
			return 1
	elif syntype == -18:  # A new notarian expression 
		if len(header) < 3:
			return 1
#		print("item[0] ===============", item[0])
#		print("header ===============", header)
		assert len(item[0][1]) == 1, "Definienda list should have unit length" + str(item[0][1])
		definiendum = item[0][1][-1]
#		print("Definiendum ===============", definiendum)
		semicolon_spot = item[0][1][-1].index(';')
		if definiendum[semicolon_spot + 1][0][0] == 43:
			scope_types = [11,15,41,43]
		elif definiendum[semicolon_spot + 1][0][0] == 42:
			scope_types = [10,14,40,42]
		if definiendum[semicolon_spot + 2][0][0] == 43:
			operand_types = [11,15,41,43]
		elif definiendum[semicolon_spot + 2][0][0] == 42:
			operand_types = [10,14,40,42]
		lasthead = header[2] 
		adjusted_len = len(item) - semicolon_spot + 3
#		print("semicolon_spot ==========", semicolon_spot)
#		print("len(item) ===============", len(item))
#		print("adjusted_len = ", adjusted_len)
#		print("lasthead ================", lasthead)
	 	# x = raw_input("Definition check")  #Find the dft dff access values
		term_or_formula = header[1][0][0]
#		print("term_or_formula =========", term_or_formula)
#		print("scope_types ==========", scope_types)
#		print("operand_types ==========", operand_types)
#		input("Pause")
		if adjusted_len < 4:      # Scope rescue section does the checking
			return 1
		elif adjusted_len == 4:
			if lasthead == 7:
				return thisheader[0] == 19  # Left scope bracket
			elif lasthead == 6:
				return thisheader[0] == 19  # Left scope bracket
			elif lasthead == 5:
				return thisheader[0] == 22  # Semi-colon
			elif lasthead == 4:
				return thisheader[0] == 21  # Colon
			elif lasthead == 3:
				if thisheader[0] in operand_types:
					pass
				else:
					return 0
#				if syntype == -6 and thisheader[0] in[10,14,40,42]:
#					pass
#				elif syntype == -7 and thisheader[0] in[11,15,41,43]:
#					pass
#				else:
#					return 0
				item[0].remove(item[0][1]) # Remove definition list
				promote(item,term_or_formula) 
				item[0][0] = item[0][0][0]
				return 1
		elif adjusted_len == 5:
			if lasthead == 7:
				return thisheader[0] == 48
			elif lasthead == 6:
				return thisheader[0] == 48
			elif lasthead == 5:
				if thisheader[0] in scope_types:
#					return 3
					return 1
				else:
					return 0
			elif lasthead == 4:
				if thisheader[0] in scope_types:
					item[0].remove(item[0][1]) # Remove definition list
					promote(item,term_or_formula) 
#					item[0][:1] = item[0][0]  # Testing??
					item[0][0] = item[0][0][0]
					return 1
		elif adjusted_len == 6:
			if lasthead == 7:
				return thisheader[0] == 22  # Semi-colon
			elif lasthead == 6:
				if thisheader[0] == 20:     # Right scope bracket
					item[0].remove(item[0][1]) # Remove definition list
					promote(item, term_or_formula)
#					item[0][:1] = item[0][0]  # Testing??
					item[0][0] = item[0][0][0]
					return 1
			elif lasthead == 5:
#				input("syntype = = = " + str(syntype))
				if thisheader[0] not in operand_types :
					return 0
#					pass
#				elif syntype == -7 and thisheader[0] in operand_types :
#					pass
#				else:
				item[0].remove(item[0][1]) # Remove definition list
				promote(item,term_or_formula) 
#				item[0][:1] = item[0][0]  # Testing??
				item[0][0] = item[0][0][0]
				return 1
		elif adjusted_len == 7:
			if thisheader[0] in scope_types:
				return 1
		elif adjusted_len == 8:
			if thisheader[0] == 20:     # Right scope bracket
				item[0].remove(item[0][1]) # Remove definition list
				promote(item, term_or_formula)
#				item[0][:1] = item[0][0]  # Testing??
				item[0][0] = item[0][0][0]
				return 1
	elif syntype in [-6,-7]:  # A notarian expression 
		lasthead = header[2] 
	 	# x = raw_input("Definition check")  #Find the dft dff access values
		term_or_formula = header[1][0][0]
		if len(item) < 4:      # Scope rescue section does the checking
			return 1
		elif len(item) == 4:
			if lasthead == 7:
				return thisheader[0] == 19  # Left scope bracket
			elif lasthead == 6:
				return thisheader[0] == 19  # Left scope bracket
			elif lasthead == 5:
				return thisheader[0] == 22  # Semi-colon
			elif lasthead == 4:
				return thisheader[0] == 21  # Colon
			elif lasthead == 3:
				if syntype == -6 and thisheader[0] in[10,14,40,42]:
					pass
				elif syntype == -7 and thisheader[0] in[11,15,41,43]:
					pass
				else:
					return 0
				item[0].remove(item[0][1]) # Remove definition list
				promote(item,term_or_formula) 
				item[0][0] = item[0][0][0]
				return 1
		elif len(item) == 5:
			if lasthead == 7:
				return thisheader[0] == 48
			elif lasthead == 6:
				return thisheader[0] == 48
			elif lasthead == 5:
				if thisheader[0] in [11,15,41,43]:
#					return 3
					return 1
				else:
					return 0
			elif lasthead == 4:
				if thisheader[0] in [11,15,41,43]:
					item[0].remove(item[0][1]) # Remove definition list
					promote(item,term_or_formula) 
#					item[0][:1] = item[0][0]  # Testing??
					item[0][0] = item[0][0][0]
					return 1
		elif len(item) == 6:
			if lasthead == 7:
				return thisheader[0] == 22  # Semi-colon
			elif lasthead == 6:
				if thisheader[0] == 20:     # Right scope bracket
					item[0].remove(item[0][1]) # Remove definition list
					promote(item, term_or_formula)
#					item[0][:1] = item[0][0]  # Testing??
					item[0][0] = item[0][0][0]
					return 1
			elif lasthead == 5:
				if syntype == -6 and thisheader[0] in[10,14,40,42]:
					pass
				elif syntype == -7 and thisheader[0] in[11,15,41,43]:
					pass
				else:
					return 0
				item[0].remove(item[0][1]) # Remove definition list
				promote(item,term_or_formula) 
#				item[0][:1] = item[0][0]  # Testing??
				item[0][0] = item[0][0][0]
				return 1
		elif len(item) == 7:
			if thisheader[0] in [11,15,41,43]:
				return 1
		elif len(item) == 8:
			if thisheader[0] == 20:     # Right scope bracket
				item[0].remove(item[0][1]) # Remove definition list
				promote(item, term_or_formula)
#				item[0][:1] = item[0][0]  # Testing??
				item[0][0] = item[0][0][0]
				return 1
	elif syntype == -14:  # Brace Expression 
		assert bracestack[-1] == '+', str(bracestack)
		print("item ========", item)
		print("braceprefix==", braceprefix)
		print("len(item) == ", len(item))
		print("thisheader == ", thisheader)
		if len(item) == 2:
			assert False
			assert thisheader[0] == 24
			if braceprefix == '\eqalign':
				braceprefix = ''
#				print("CCCCCCCCCCCCCCCCCCCCCCCCCCC")
				promote(item, 61)
			elif braceprefix == '\sqrt':
#				print("item =======", item)
#				input("Hold it")
				pass
			return 1
		elif len(item) == 3:
			if  thisheader[0] == 26:  #Font modifier
				assert False
				x = bracestack.pop()
				assert x == '+',"Brace stack pop yielded: " + x
				bracestack.append(item[-1][1])
				current_fontmods = current_fontmods | frozenset([item[-1][1]])
				n = fontmoddict.get(current_fontmods, -1)
				if n == -1:
					n = len(fontmoddict) + 1
					fontmoddict[current_fontmods] = n
				current_fontnum = n
#				print("current_fontnum =========", current_fontnum)
				promote(item, 61)
			elif thisheader[0] == 10:
				assert False
				pass
			else:
				print("braceprefix =====", braceprefix)
				print("item ========", item)
				print("thisheader[0] ====", thisheader[0])
				input("Hold it")
				assert False
				pass
			return 1
		elif len(item) == 4:
			if item[1][0][0] != 24: 
				return 0
			if thisheader[0] == 4:
#				item[:] = item[2]
				promote(item, 40)
				return 1
			else:
				return 0
		else:
			print
			print("No brace expressions yet")
			assert False
				
	return 0


def deflistupdate(item):
	deflist = item[0][1]
#	print "Number of defs: ", len(deflist)
	n = len(item) - 1
	a = []
	for definiendum in deflist:
#		definiendum = d[1]
		ibvlist = definiendum[0][1][1:]
		if n in ibvlist:
			if item[-1][0][0] == 10:
				a.append(definiendum)
		elif syntmatch(definiendum[n], item[-1]):
			a.append(definiendum)
	item[0][1] = a
	return len(a)

def syntmatch(form, instance):
	headeri = instance[0]
	arity = mathdb[MD_ARITY]
	if type(form) is str :
		if symtype(form) == 10: 
# This should never happen.
			return headeri[0] in [10,14,40,42] 
		elif symtype(form) == 11:
			return headeri[0] in [11,15,41,43] 
		else:
			return form == instance[1]
	else:
		headerf = form[0]
		if headerf[0] == 42:
# This should never happen.
			return headeri[0] in [10,14,40,42] 
		if headerf[0] == 43:
# This should never happen.
			return headeri[0] in [11,15,41,43] 
		if headerf[0] in [40,41]:
			indvars = indvlist(form)
			for n in range(1,len(form)):
				ok = subsyntmatch(form[n],instance[n],indvars)
				if not ok:
					return 0
			return 1

def subsyntmatch(form, instance,indvars):
	headeri = instance[0]
	arity = mathdb[MD_ARITY]
	if type(form) is str: 
		if symtype(form) == 10: 
			if form in indvars:
				return headeri[0] == 48 
			elif type(instance) is str: 
				return symtype(instance) in [10,14]
			else:
				return headeri[0] in [10,14,40,42] 
		elif symtype(form) in [11,12,13] and arity[form] == 0: # form is a schemator
			return headeri[0] in [11,15,41,43] 
		else:
			return form == instance
	else:
		headerf = form[0]
		if headerf[0] == 42:
			return headeri[0] in [10,14,40,42] 
		if headerf[0] == 43:
			return headeri[0] in [11,15,41,43] 
		if headerf[0] in [40,41]:
			for n in range(1,len(form)):
				ok = subsyntmatch(form[n],instance[n],[])
				if not ok:
					return 0
			return 1

def paradecrop(item, reducedmax):
#	print("paradecrop item: =====", item)
	header = item[0]
	while reducedmax < header[1]:#compare with parade's max
		thisprec = header[1]
		newitem = []
		x = item.pop()
		if x[0][0] in [1,2,3]:
			lastprec = x[2]
		while x[0][0] not in [1,2,3] or x[2] >= thisprec:
			newitem.append(x)
			if len(item) == 1:
				newitem.reverse()
				newitem = [[44, thisprec]] + newitem
#				input("paradecrop " + str(newitem))
				a = paradecheck(newitem)
#				input("a ====" + str(a))
				if a == 0:
					promote(newitem, 50)
					newitem[0][0] = 50
					item.append(newitem)
					return "Error: Parade syntax"
				elif type(a) is str:
					promote(newitem, 50)
					newitem[0][0] = 50
					item.append(newitem)
					return a 
				promote(newitem, a)
				item.append(newitem)
				return 1
			x = item.pop()
			if x[0][0] in [1,2,3]:
				lastprec = x[2]
		if x[0][0] in [1,2,3]:
			item.append(x)
		item[0][1] = lastprec 
		newitem.reverse()
		newitem = [[44, thisprec]] + newitem
		a = paradecheck(newitem)
		if a == 0:
			newitem[0][0] = 50
			item.append(newitem)
			return "Error: Parade syntax"
		elif type(a) is str:
			newitem[0][0] = 50
			item.append(newitem)
			return a 
		promote(newitem, a)
		if len(item) > 1:
			item.append(newitem)
		else:
			item.append(newitem)
			break
	return 1


def paradecheck(item):
	if len(item)  == 1: 
		return "Empty parade"
	prec = item[0][1]

 #Check for variations in the precedence 
	assert all([x[2] == prec for x in item[1:] if x[0][0] in [1,2,3]])

#	n = len(item)
#	for i in range(1, n - 1):
#		if item[i][0][0] in [1,2,3] and item[i + 1][0][0] in [1,2,3]:
#			return "Nexus " + item[i][1] + " " + item[i+1][1] + " not declared"

	if prec in mathdb[MD_CTCHK]:
		checker = eval(mathdb[MD_CTCHK][prec])
		retval = checker(item)
		if not(retval == 0 or type(retval) is str):
			return retval 
		else:
			print(item, "rejected by ",mathdb[MD_CTCHK][prec])
			if type(retval) is str:
				print(retval)
			return retval
	else:
		needed_syntax_handlers.add(prec)
#		print("No syntax handler defined for precedence level " +  str(prec))
#		return "TEST"
		return deflistcheck(item)

def deflistcheck(item):
#For parenthetical expressions only.
	defs = mathdb[MD_CTDEFS]
	for x in item[1:]:
		if x[0][0] in [1,2,3]:
			first_operator = x[1]
			break
	else:
		first_operator = "Adjacency"
	if first_operator not in defs:
		return "unknown operator: " + first_operator
	for defn in defs[first_operator]:
		ctm = ctmatch(defn,item)
		if ctm: 
			return ctm 
#	print("first_operator ==============", first_operator)
#	print("defs[first_operator]=========", defs[first_operator])
	return "No definition for: " + str(item)

def ctmatch(defn, instance):
	assert type(defn) is list and type(instance) is list	
	if len(defn) != len(instance):
		return False
	for j in range(1,len(defn)):
		if symtype(defn[j]) == 10:
			if type(instance[j]) is str:
				if symtype(instance[j]) not in [10,14]:
					return False
			elif instance[j][0][0] not in [10,14,40,42,44]:
				return False
		elif symtype(defn[j]) == 11:
			if type(instance[j]) is str:
				if symtype(instance[j]) not in [11,15]:
					return False
			elif instance[j][0][0] not in [11,15,41,43,45]:
				return False
		elif defn[j] != instance[j][1]:
			return False
	return defn[0][0] 

def mix_check(item):
	return n_arycheck(item, just_one_repeated_op = False)

def dontcheck(item):
	return str(item) + " is not a true infix operator."

def binarycheck(item):
	alc = n_ary_altopcheck(item)
	if type(alc) is str: 
		return alc
	elif len(item) != 4:
		return item[2][1] + " is binary only."
	else:
		return alc

def n_ary_altopcheck(item):
	All_terms = True
	All_formulas = True
	last_op = None
	for i,x in enumerate(item):
		assert type(x) is list
		if i == 0:
			pass
		elif i % 2 == 1 and x[0][0] in [10,14,40,42,44]:
			All_formulas = False
		elif i % 2 == 1 and x[0][0] in [11,15,41,43,45]:
			All_terms = False
		elif i % 2 == 1:
			return "Term or formula expected instead of " + x[1]
		elif i % 2 == 0 and last_op == None: 
			last_op = x[1]
		elif i % 2 == 0 and last_op != x[1]: 
			return "ops differ: "+last_op +","+ x[1]	
	if not All_formulas and not All_terms:
		return "Terms and formulas mixed."
	if last_op not in mathdb[MD_CTDEFS]:
		return "No definition for " + last_op
	bindeflist = mathdb[MD_CTDEFS].get(last_op)
	if not bindeflist:
		return "No definition for " + last_op

	for y in bindeflist:
		if len(y) == 4: 
			bindef = y
			return bindef[0][0]
	else:
		return "No binary definition for " + last_op

def n_arycheck(item, just_one_repeated_op = True):
#	print("n_arycheck item: ====", item)
	n = len(item)
	if n == 3:
		return "Unary usage not allowed: " + str(item)
	if n == 2:
		return "Solitary operator not allowed: " + str(item)
	assert not n < 2, str(item)
	All_terms = True
	All_formulas = True
	first_op = None
	for i in range(1,n):
		assert type(item[i]) is list
		if item[i][0][0] in [10,14,40,42,44]:
			All_formulas = False
		elif item[i][0][0] in [11,15,41,43,45]:
			All_terms = False
		else:
			if not first_op:
				first_op = item[i][1]
			elif first_op != item[i][1] and just_one_repeated_op:
				return "ops differ: "+first_op +","+item[i][1]	

	bindeflist = mathdb[MD_CTDEFS].get(first_op)
	if not bindeflist:
		return "No definition for " + first_op

	for x in bindeflist:
		if len(x) == 4:
			bindef = x
			break
	else:
		return "No binary definition for " + first_op

	if (All_terms and bindef[0][0] == 44) or (All_formulas and bindef[0][0] == 45):
		return bindef[0][0] 
#	elif same_or_last == 1 and bindef[0][0] in [44,45]:
#		return bindef[0][0]
	else:
		print("item====", item)
		print("bindef=====", bindef)
		print("bindef[0][0]==", bindef[0][0])
		return "Mixed types for " + first_op

def Morse_verbcheck(item):
	for x in item[1:]: 
		if x[0][0] in  [1, 2, 3]:
			last_op = x[1]
#	if last_op == ',':
#		ncheck = n_arycheck(item)
#		if type(ncheck) is str:
#			return ncheck
#		else:
#			return 44  ########### Comma
	binvcheck = binverbcheck(item)
	if type(binvcheck) is str:
		return binvcheck
	elif last_op in mathdb[MD_CTDEFS]:
		return mathdb[MD_CTDEFS][last_op][0][0][0] #Fetch return type, 44 or 45
	else:
		return "new operator: " + last_op
	
def verbcheck(item):
	for x in item[1:]: 
		if x[0][0] in  [1, 2, 3]:
			last_op = x[1]
	if last_op == ',':
		ncheck = n_arycheck(item)
		if type(ncheck) is str:
			return ncheck
		else:
			return 44  ########### Comma
	binvcheck = binverbcheck(item)

	if type(binvcheck) is str:
		return binvcheck
	elif last_op in mathdb[MD_CTDEFS]:
		return mathdb[MD_CTDEFS][last_op][0][0][0] #Fetch return type, 44 or 45
	else:
		return "new operator: " + last_op
	
def complexverbcheck(item):
	pass

def commapluralcheck(item):
#
# values of state variable:
#    0  subject
#    1  verb
#    2  object
# 
 
	state = 0
	j = 0
  
	for x in item[1:]:
		j = j + 1
		y = x[0][0]
		if state == 0:
			if y == 3 or y == 1:
				if j == 1:
					return "Term needed instead of "  + str(x[1])
				elif pspeech(x[1]) == 'verb':
					state = 1
				elif j % 2 == 1:
					return "Verb or term needed. Got: " + str(x[1])
				elif j % 2 == 0 and pspeech(x[1]) == 'conjunction':
					pass	
				else:
					return "Verb or conjunction needed instead of " + str(x[1])
			elif y in [10,14,40,42,44]:
				if j % 2 == 0: 
					return "Got " + str(x[1]) + "in place of a connector"
			else:
				return "Term or infix operator needed.  Got " + str(x[1])
		elif state == 1:
			if y in [10,14,40,42,44]:
				state = 2
				k = j  #Grab parity
			elif y != 3:
				return "Bad Nexus."
		elif state == 2:
			if y == 3 or y == 1:
				if (j - k) %2 == 0:
					return "Term needed. Got: " + str(x[1])
			elif y in [10,14,40,42,44]:
				if (j - k) %2 == 1:
					return "Verb needed. Got: " + str(x[1])
			else:
				return "Term of infix operator needed. Got " + str(x[1])
	if state == 0:
		return 44 
	elif state == 2:
		return 45 
	else:
		assert False, str(item)
		return  "Incomplete statement."


def binverbcheck(item):
#
# values of state variable:
#    0  subject
#    1  verb
#    2  object
# 
 
	state = 0
	j = 0
  
	for x in item[1:]:
		j = j + 1
		y = x[0][0]
		if state == 0:
			if y == 3 or y == 1:
				if j == 1:
					return "Start with a term." 
				elif j % 2 == 1:
					return "Term needed." 
				elif pspeech(x[1]) == 'verb':
					state = 2
				elif pspeech(x[1]) == 'conjunction':
					pass	
				else:
					return "Verb error."
			elif y in [10,14,40,42,44]:
				if j % 2 == 0: 
					print("Rescue code should catch this")
					raise SystemExit
#				else: This is the indicial variable case
			else:
				return "Error: Term needed." 
#		elif state == 1:
#			if y in [10,14,40,42,44]:
#				state = 2
#			elif y != 3:
#				return "Bad Nexus."
		elif state == 2:
			if y == 3 or y == 1:
				if pspeech(x[1]) == 'reset':
					state = 0
			elif y not in [10,14,40,42,44]:
				return "Error: term needed"
	if state == 2:
		return True 
	else:
		assert False, str(item)
		return  "Incomplete statement."


def followercheck(item):
#
	j = 0
  
	for x in item[1:]:
		j = j + 1
		y = x[0][0]
		if j == 1:
			if y in [10,14,40,42,44]:
				pass
			else:
				return "followercheck: Term missing"
		elif y == 3:
			pass
		else:	
			print("item ========== ", item)
			return "followercheck: " + str(x) + "Not a follower"
	return 44 


def mixcheck(item):
	n = len(item)
	if n < 3:
		return "+- error"
	if item[1][0][0] == 3:
		if item[2][0][0] not in  [10,14,40,42,44]:
			return "+- error"
	for i in range(1,n -2):
		if item[i][0][0] in [10,14,40,42,44]:
			if item[i+1][0][0] != 3:
				return "+- error"
			if item[i+2][0][0] not in [10,14,40,42,44]:
				return "+- error"
		elif item[i][0][0] != 3:
			return "+- error"
	return 44 # Term

def condelscheck(item):
	n = len(item)
	if n < 4 or n % 2 == 1:
		return "Even length for cond-els"
	for k in range(1,n,2): # All terms or formulas
		if item[k][0][0] in [10,14,40,42,44]:
			pass
		elif item[k][0][0] in [11,15,41,43,45]:
			pass
		else:
			return str(item[k]) + " not a term or formula in cond-els"
	return 44
	for k in range(2,n,2):
		if item[k][1] != '\\els':
			break
	else:
		for k in range(1,n,2): # All terms
			if item[k][0][0] not in [10,14,40,42,44]:
				return 0
		return 1
	for k in range(2,n,2):
		if (k//2)%2 == 1:
			if item[k][1] != '\\cond':
				return 0
			if item[k-1][0][0] not in [11,15,41,43,45]:
				return 0
		else:
			if item[k][1] != '\\els':
				return 0
			if item[k-1][0][0] not in [10,14,40,42,44]:
				return 0
	if item[n-1][0][0] not in [10,14,40,42,44]:
		return 0
	return 1

def pspeech(x):
	psp = mathdb[MD_PSPEECH]
	prec= mathdb[MD_PRECED]
	if x in psp:
		return psp[x]
	elif x in prec and prec[x] in psp: 
		return psp[prec[x]]

def scopecheck(item):    #Old version
	if len(item)  == 1: 
		print("Error: Empty scope.")
		return 0
#
# values of state variable:
#    0  subject
##   1  verb
#    2  object
#
	state = 0
	j = 0
# 
	for x in item[1:]:
		j = j + 1
		y = x[0][0]
		if state == 0:
			if y == 3:
				if j % 2 == 1:
					print("Error: Indicial variable needed.")
					return 0
				elif pspeech(x[1]) == 'verb':
					state = 2
				elif pspeech(x[1]) == 'conjunction':
					pass
				elif pspeech(x[1]) == 'reset':
					print("reset symbol", x[1], "not allowed in subject.")
					return 0
				else:
					print("Error: verb or conjunction needed.")
					return 0
			elif y == 10:
				if j % 2 == 0: 
					raise RuntimeError("Rescue code should catch this")
#				else: This is the indicial variable case
			else:
				print("Error: Indicial variable needed.")
				return 0
#		elif state == 1:
#			if y in [10,14,40,42,44]:
#				state = 2
#			elif y != 3:
#				print("Error: Bad nexus.")
#				return 0
		elif state == 2:
			if y == 3:
				if j % 2 == 1:
					print("Error: Term needed.")
					return 0
				if pspeech(x[1]) == 'reset':
					state = 0
			elif y in [10,14,40,42,44]:
				if j % 2 == 0: 
					raise RuntimeError("Rescue code should catch this")
			else:
				print("Error: Term needed.")
				return 0
	return 1 

def scopecheck(item):   #New version
	if len(item)  == 1: 
		print("Error: Empty scope.")
		return 0
#
# values of state variable:
#    0  subject
##   1  verb
#    2  object
#
	state = 0
	j = 0
# 
	for x in item[1:]:
		j = j + 1
		y = x[0][0]
		if state == 0:
			if y == 3:
				if j == 1:
					print("Error: Indicial variable needed. Got ", x[1])
					return 0
				elif pspeech(x[1]) == 'verb':
					state = 1
				elif j % 2 == 1:
					print("Verb or indicial variable needed. Got ", x[1])
					return 0
				elif j % 2 == 0 and pspeech(x[1]) == 'conjunction':
					pass
				elif pspeech(x[1]) == 'reset':
					print("reset symbol", x[1], "not allowed in subject.")
					return 0
				else:
					print("Verb or conjunction needed.   Got", x[1])
					return 0
			elif y == 10:
				assert j % 2 == 1,"Rescue code should catch this"
#				else: This is the indicial variable case
			else:
				print("Indicial variable or infix operator needed.  Got ",x[1])
				return 0
		elif state == 1:
			if y in [10,14,40,42,44]:
				state = 2
				k = j  # Grab parity
			elif y != 3:
				print("Error: Bad nexus symbol:", x[1])
				return 0
		elif state == 2:
			if y == 3:
				if (j - k) % 2 == 0:
					print("Error: Term needed. not", x[1])
					return 0
				if pspeech(x[1]) == 'reset':
					state = 0
			elif y in [10,14,40,42,44]:
				assert (j - k) % 2 == 0, "Rescue code should catch this"
				pass
			else:
				print("Term or infix operator needed. Got", x[1])
				return 0
	return 1 
			
def definitioncheck(item):
	definiendum = item[1]
	definor = item[2]
	definiens = item[3]
#	print("definitioncheck: ")
	assert type(definor) is list and type(definor[0]) is list and definor[0][0] in [1,2], "No definor!" 
	if definor[0][0] == 1: 
		if definiens[0][0] not in [10, 14, 40,42]:  # A Term 
			return "Error: Type mismatch, term expected" 
	elif definor[0][0] == 2:
		if definiens[0][0] not in [11, 15, 41,43]:  # A formula 
			return "Error: Type mismatch, formula expected"
	register = register_definiendum(definiendum,definor[0][0])
	if type(register) is str:
		return register	
	bvlist = set([])
	if symtype(definiendum[1]) in [8,9]:
		for x in definiendum[1:]:
			if type(x) is list and x[0][0] in [42,43]:
					bvlist = set(x[2:])
					break
		else:
			assert False 

#	print('definiendum =',definiendum)
#	print('definor =',definor)
#	print('definiens =', definiens)
	left_vars = set(nblist(definiendum)) - bvlist
	right_vars = set(nblist(definiens))
	if right_vars - left_vars:
		print("left_vars", left_vars)
		print("right_vars", right_vars)
		error_message = "Suppressed variable(s): " 
		for x in (right_vars - left_vars):
			error_message = error_message + " " + x
		return error_message 
	elif left_vars - right_vars:
		error_message = "Unused variable(s): " 
		for x in (left_vars - right_vars):
			error_message = error_message + " " + x
		print("left_vars", left_vars)
		print("right_vars", right_vars)
		return error_message 
	else:
		return 1
	
def register_definiendum(definiendum,termorformula):
	#termorformula = 1, for TERM
	#termorformula = 2, for FORMULA
	td = mathdb[MD_SYMTYPE]
	precedence = mathdb[MD_PRECED]
	defs = mathdb[MD_DEFS]
	introductor = definiendum[1] 

	if symtype(introductor) == 16: # Left parenthesis
		if definiendum[2][0][0] != 50:
			return "Bad definiendum:"  + str(definiendum)
		if definiendum[3]!= ')':
			return "Bad definiendum:"  + str(definiendum)
		definiendum = definiendum[2]
		if termorformula == 1:
			definiendum[0][0] = 44
		else:
			definiendum[0][0] = 45

#		print("DEFINIENDUM1 =", definiendum)
		first_op = None
		for x in definiendum[1:]:
			if type(x) is str :
				if symtype(x) not in [10,11]:
					if not first_op:
						first_op = x
						if symtype(x) not in [1,2,3]:
							return "Precedence of infix operator " + x + " not set."
						p = precedence[x]
					elif symtype(x) == 5:
						return x + " must be assigned a precedence."
					elif symtype(x) == 7:
						return x + " must be assigned a precedence."
					elif symtype(x) == 17:
					 	return x + " already used as an introductor in: " + str(defs[x][0])
					elif symtype(x) not in [1,2,3]:
						return x + " used as an infix operator."
					elif p != precedence[x]:
						return "Conflicting precedence values," + \
						 first_op + ":"+str(p)+ "; " + x + ":" + str(precedence[x])

		defs = mathdb[MD_CTDEFS]  
			#  Dictionary storing a list of definienda for connectors
			#  Need to check for conflicts!
		if first_op and first_op in defs:
			definiendum[0][1] = [len(defs[first_op])-1]
			if definiendum not in defs[first_op]:
				defs[first_op].append(definiendum)
		elif first_op:
#			print("storing definiendum ================", definiendum)
#			print("first_op ===", first_op)
#			definiendum[0][1] = [0]
			defs[first_op]= [definiendum]
		if len(definiendum) == 4:
				pass
		else:
				print("Warning: Non-binary connector")
				pass
#		print("DEFINIENDUM3 =", definiendum)
#		elif len(definiendum) == 6:
#				return "Error: Non-binary connector with programmed precedence!"
	elif symtype(introductor) in [28,29]:  # A new notarian
		new_definiendum = flatten(definiendum)
		definiendum = new_definiendum
		if termorformula == 1: # A term
			definiendum[0][0] = 40
		else:                  # A formula 
			definiendum[0][0] = 41
		bvlist = []
		varlist = []
		schematorlist = []
		schemexplist = []
		term_schemexp = []
		formula_schemexp = []
		scope_schemexp = []
		semicolon_spot = 0 
		spot = 2
		for x in definiendum[2:]:
			if type(x) is str: 
				if symtype(x) == 5:
						td[x] = 7
				elif symtype(x) in [10,11]:
					if x in varlist:
						return "Error: Repeated occurrence of " + x
					else:
						varlist.append(x)
				elif symtype(x) == 22:
					semicolon_spot = spot
			elif x[0][0] in [42,43]:
				if x[1] in schematorlist:
					return "Error: " + x[1] +  "repeated schemator in definiendum"
				else:
					schematorlist.append(x[1])
					schemexplist.append(x)
				if spot - 1 == semicolon_spot:
					scope_schemexp = x 
				for y in x[2:]:
					if type(y) is not str or symtype(y) != 10:
						return "Error: Non-variable " + str(y) + "not allowed after a schemator"
					elif y not in bvlist:
						bvlist.append(y)
			else:
				assert False, str(definiendum)
			spot = spot + 1 
		for y in bvlist:
			if y not in varlist:
				return "Error: Indicial position missing for: " +  y
		if bvlist:
			for x in schemexplist:
				if x != x[:2] + bvlist:
					return str(x) + " has a different bound variable."
		else:
			return "Notarian without a bound variable!"
		if definiendum[semicolon_spot -1: semicolon_spot + 2] != bvlist + [';'] + [scope_schemexp]:
			print("definiendum ====== ", definiendum)
			print("bvlist + scope =====",  bvlist + [';'] + [scope_schemexp])
			return str(definiendum) + "not in 'boundvar;scope' form"
		print("Definiendum ====== ", definiendum)
		if introductor in defs:
			#Compatibility check needed here
			defs[introductor].append(definiendum)
		else:
			defs[introductor] = [definiendum]  # Needed to determine T or F status
	elif symtype(introductor) in [17,7,5]:  # An introductor which is not a parenthesis
		new_definiendum = flatten(definiendum)
		definiendum = new_definiendum
		if termorformula == 1: # A term
			definiendum[0][0] = 40
		else:                  # A formula 
			definiendum[0][0] = 41
		bvlist = []
		varlist = []
		schematorlist = []
		term_schemexp = []
		formula_schemexp = []
		for x in definiendum[2:]:
			if type(x) is str: 
				if symtype(x) == 5:
						td[x] = 7
				elif symtype(x) in [10,11]:
					if x in varlist:
						return "Error: Repeated occurrence of " + x
					else:
						varlist.append(x)
			elif x[0][0] in [42,43]:
				if x[1] in schematorlist:
					return "Error: " + x[1] +  "repeated schemator in definiendum"
				else:
					schematorlist.append(x[1])
				if x[0][0] == 42:
					term_schemexp = x
				else:
					formula_schemexp = x
				for y in x[2:]:
					if type(y) is not str or symtype(y) != 10:
						return "Error: Non-variable " + str(y) + "not allowed after a schemator"
					elif y not in bvlist:
						bvlist.append(y)
			else:
				assert False, str(definiendum)
		for y in bvlist:
			if y not in varlist:
				return "Error: Indicial position missing for: " +  y
		if bvlist:
			initialsegment =   [introductor, bvlist[0]]
			for y in bvlist[1:]:
				initialsegment = initialsegment + [',', y]
		if term_schemexp and formula_schemexp and\
            term_schemexp[2:] == formula_schemexp[2:] == bvlist and\
				definiendum[1:] == initialsegment + [';', formula_schemexp, term_schemexp]:
			if symtype(introductor) == 5:
				td[introductor] = 8
				defs[introductor] = [definiendum]  # Needed to determine T or F status
			else:
				return "Error: Defining a known constant as a notarian not allowed." 
		elif formula_schemexp and formula_schemexp[2:] == bvlist and\
            definiendum[1:] == [introductor] + bvlist + [formula_schemexp]:			
			if symtype(introductor) == 5:
				td[introductor] = 9
				defs[introductor] = [definiendum]  # Needed to determine T or F status
			else:
				return "Error: Defining a known constant as a notarian not allowed." 
		else:
			ibvlist = []
			for k in range(len(definiendum)):
				if definiendum[k] in bvlist:
					ibvlist.append(k)
			if symtype(introductor) == 5:
				if len(definiendum) == 2:
					if termorformula == 1: 
						td[introductor] = 14
					else:
						td[introductor] = 15
				else:
					definiendum[0][1:] = [[0]]
					definiendum[0][1].extend(ibvlist)
					defs[introductor] = [definiendum]
					td[introductor] = 17
			elif symtype(introductor) == 7:
					clash = clash_check(introductor)
					if type(clash) is str:
						return clash
					definiendum[0][1:] = [[0]]
					definiendum[0][1].extend(ibvlist)
					defs[introductor] = [definiendum]
					td[introductor] = 17
			elif symtype(introductor) == 17:
#           If definiendum is subsumed it just parses.
				clash = defclash(definiendum)
				if type(clash) is str:
						return clash
				definiendum[0][1:] = [[len(defs[introductor])]]
				definiendum[0][1].extend(ibvlist)
				defs[introductor].append(definiendum)
#				introductor of parsed_exp = parsed_exp[1]  
#				definiendum of parsed_exp = defs[introductor][parsed_exp[0][1][0]]
#				definiendum = defs[introductor]
#				ibvlist of definiendum = definiendum[0][1][1:]
	else:  # An introductor which is not a parenthesis
		return introductor + " not allowed as an initial symbol of a form."
	return 1

def flatten(exp):
	assert type(exp) is list
	return exp[:1] + decap_flatten(exp)

def decap_flatten(exp):
		retval = []
		for x in exp[1:]:
			if type(x) is str:
				retval.append(x)
			elif type(x) is list and x[0][0] in [42,43]:
				retval.append(x)
			else:
				retval.extend(decap_flatten(x))
		return retval

def clash_check(new_intro):
	defs = mathdb[MD_DEFS]
	for old_intro in defs:
		deflist = defs[old_intro]
		for i in range(len(deflist)):
			for j in range(i + 1, len(deflist)):
				k = 1
				while sigmatch(deflist[i][k], deflist[j][k]): 
					k = k + 1
				if deflist[i][k]==new_intro and \
					(sigtermp(deflist[j][k]) or formulap(deflist[j][k])):
						return new_intro + " clashes with " + str(deflist[i])
				if deflist[j][k]==new_intro and \
					(sigtermp(deflist[i][k]) or formulap(deflist[i][k])):
						return new_intro + " clashes with " + str(deflist[j])
	return 1

def defclash(new_definiendum):
	introductor = new_definiendum[1]
	deflist = mathdb[MD_DEFS][introductor]
	for j in range(len(deflist)):
		k = 1
		while k < min(len(new_definiendum),len(deflist[j])):
			if sigmatch(new_definiendum[k], deflist[j][k]): 
				k = k + 1
			else:
				break
		else:
			return str(new_definiendum) + " clashes with " + str(deflist[j])
		if k == len(new_definiendum):
				return new_intro + " clashes with " + str(deflist[j])
		if k == len(deflist[j]):
				return str(new_definiendum) + " clashes with " + str(deflist[j])
		if type(deflist[j][k]) is str and \
			 symtype(deflist[j][k]) in [8,9,14,15,17] and \
			(sigtermp(new_definiendum[k]) or sigformulap(new_definiendum[k])):
				return str(new_definiendum) + " clashes with " + str(deflist[j])
		if type(new_definiendum[k]) is str and \
			 symtype(new_definiendum[k]) in [8,9,14,15,17] and\
			(sigtermp(deflist[j][k]) or sigformulap(deflist[j][k])):
				return str(new_definiendum) + " clashes with " + str(new_definiendum)
	return False

	
def sigtermp(exp):
	if type(exp) is str:
			return symtype(exp) == 10
	else:
			return exp[0][0] == 42

def sigformulap(exp):
	if type(exp) is str:
			return symtype(exp) == 11
	else:
			return exp[0][0] == 43

def sigmatch(item1, item2):
	if item1 == item2:
		return True
	if sigtermp(item1) and sigtermp(item2):
		return True
	if sigformulap(item1) and sigformulap(item2):
		return True
	return False

def trace_fun(string):
	retlist = []
	num = pattern.nums 
	while string:
		string = string.lstrip()
		if string:
			numm = num.match(string)
			if numm:
				retlist.append(int(numm.group(1)))
				string = string[numm.end(1):]
			else:
				retlist.append(string[0])
				string = string[1:]
	return retlist
			
def process_directive(comment_line, hereditary_only=True):
	directivem = pattern.directive.match(comment_line)
	tokenp = pattern.token
	if directivem.group(1) == 'set_precedence:':
		tail = directivem.group(2).split()
		if len(tail) < 2:
			return "Bad precedence assignment"
		n = tail.pop()
		if not n.isdigit(): 
			return "Numerical precedence value needed," + " not " + n
		n = int(n)
		middle = " ".join(tail)
		nexus = [m.group(2) for m in tokenp.finditer(middle)]
		if len(nexus) == 1:
			connector = nexus[0]
			if mathdb[MD_SYMTYPE].get(connector, -1) == -1:
				mathdb[MD_SYMTYPE][connector] = 3 
			elif mathdb[MD_SYMTYPE][connector] in [7]:
				mathdb[MD_SYMTYPE][connector] = 3 
			elif mathdb[MD_SYMTYPE][connector] in [1,2,3]:
				pass
			else:
				return "Setting precedence on " + connector + " not allowed."
		else:
			for connector in nexus:
				if mathdb[MD_SYMTYPE].get(connector, -1) not in [1,2,3]:
					return "Nexus " + str(nexus) + " contains unknown connector: " + connector 
#		new_nexus = " ".join(nexus)
#			if connector in mathdb[MD_PRECED]:
#				if mathdb[MD_PRECED][connector] == n:
#					pass
#				else:
#					print("Changing old precedence of",connector,"from",
#												mathdb[MD_PRECED][connector],"to",n) 
		if len(nexus) > 1:	
			bigtoken = " ".join(nexus)
			mathdb[MD_SYMTYPE][bigtoken] = 3
			mathdb[MD_PRECED][bigtoken] = n
		else:
			mathdb[MD_PRECED][nexus[0]] = n
		return 1 
	elif directivem.group(1) == 'speechpart:':
		psp = mathdb[MD_PSPEECH]
		tail = directivem.group(2).split()
		kind = tail.pop()
		for item in tail:
			if item.isdigit():
				if int(item) in psp and psp[int(item)] != kind:
					return "Precedence level " +  item +  "already assigned to " +  pseech[int(item)]
				else:
					psp[int(item)] = kind
			else:
				if item in psp and psp[item] != kind:
					return item +  "already assigned to " +  psp[item]
				else:
					psp[item] = kind
		return 1 
	elif directivem.group(1) == 'variable:':
		tail = directivem.group(2).split()
		for item in tail:
			mathdb[MD_SYMTYPE][item] = 10 
		return 1 
	elif directivem.group(1) == 'flanker:':
		tail = directivem.group(2).split()
		for item in tail:
			mathdb[MD_SYMTYPE][item] = 30 
		return 1 
	elif directivem.group(1) == 'identity:':
		tail = directivem.group(2).split()
		if len(tail) == 0:
			return "No operator given"
		if len(tail) > 1:
			return "One operator only"
		new_id = tail[0]
		if new_id not in mathdb[MD_IDOPS]:
			mathdb[MD_IDOPS].append(new_id)
		return 1
	elif directivem.group(1) == 'associative:':
		tail = directivem.group(2).split()
		if len(tail) == 0:
			return "No operator given"
		n = 0 
		if tail[-1].isdigit():
			n = int(tail.pop())
			if n < 1:
				return "Invalid line number"
		tail = " ".join(tail)
		nexus = [m.group(2) for m in tokenp.finditer(tail)]
		op = " ".join(nexus)
		if op == '':
			return "No operator given"
		if op not in mathdb[MD_PRECED]:
				return op + " is not a known connector"
		if op not in mathdb[MD_PROPGET]:
				mathdb[MD_PROPGET][op] = {}
		if n:
				mathdb[MD_PROPGET][op]['associative'] = n
		else:
				mathdb[MD_PROPGET][op]['associative'] = True
		return 1 
	elif directivem.group(1) == 'commutative:':
		tail = directivem.group(2).split()
		if len(tail) == 0:
			return "No operator given"
		n = 0 
		if tail[-1].isdigit():
			n = int(tail.pop())
			if n < 1:
				return "Invalid line number"
		tail = " ".join(tail)
		nexus = [m.group(2) for m in tokenp.finditer(tail)]
		op = " ".join(nexus)
		if op == '':
			return "No operator given"
		if op not in mathdb[MD_PRECED]:
				return op + " is not a known connector"
		if op not in mathdb[MD_PROPGET]:
				mathdb[MD_PROPGET][op] = {}
		if n:
				mathdb[MD_PROPGET][op]['commutative'] = n
		else:
				mathdb[MD_PROPGET][op]['commutative'] = True
		return 1 

	elif directivem.group(1) == 'transitive:':
		tail = directivem.group(2)
		filerangep = re.compile("(\S+)(?:\s+(\d*)[\,\-]\s*(\d*))?")
		filerangem = filerangep.match(tail)
		if not filerangem:
			return "Invalid transitive directive" 
		if filerangem.group(3):
			entry = [filerangem.group(1), int(filerangem.group(2)), int(filerangem.group(3))]
		elif filerangem.group(2):
			entry = [filerangem.group(1), int(filerangem.group(2)), int(filerangem.group(2))]
		else:
			entry = [filerangem.group(1)]
		if entry not in mathdb[MD_TRFILE]:
			mathdb[MD_TRFILE].append(entry)
		return 1
		
	elif directivem.group(1) == 'def_precedence:':
#		print("Define checker for a precedence level" )
#		print(directivem.group(3),directivem.group(4).rstrip())
#		n = directivem.group(3)
#		mathdb[MD_CTCHK][n] = directivem.group(4).rstrip()
		tail = directivem.group(2).split()
		if len(tail) != 2 and len(tail) != 3:
			return "Bad precedence definition"
		n = tail[0]
		if not n.isdigit():
				return "precedence level must be an integer"
		n = int(n)
		mathdb[MD_CTCHK][n] = tail[1].rstrip()
		if len(tail) == 3:
			mathdb[MD_CTHNDL][n] = tail[2].rstrip()
		
	elif directivem.group(1) == 'def_symbol:':
#		mathdb[MD_MACR][directivem.group(3)[1:]] = directivem.group(4)[1:].rstrip()
		mathdb[MD_MACR][directivem.group(3)] = directivem.group(4).rstrip()
#		print(directivem.groups())
#	elif directivem.group(1) == 'external_ref:' and not hereditary_only:
	elif directivem.group(1) == 'external_ref:':
		mathdb[MD_REFD][directivem.group(3)] = directivem.group(4).rstrip()
	elif directivem.group(1) == 'major_unit:':
		# Handle this in renum.
		pass
	elif directivem.group(1) == 'subfile:':
		# Handle this in makedf.
		pass
	elif directivem.group(1) == 'term_definor:':
		token = directivem.group(2).strip()
		if tokenp.match(token).group(2) != token: 
			return "Can't use " + token + " as a definor symbol."
		mathdb[MD_SYMTYPE][token] = 1 
		if not mathdb[MD_PRECED].get(token):
			return "Definor needs a precedence value."
	elif directivem.group(1) == 'formula_definor:':
		token = directivem.group(2).strip()
		if tokenp.match(token).group(2) != token: 
			return "Can't use " + token + " as a definor symbol."
		mathdb[MD_SYMTYPE][token] = 2 
		if not mathdb[MD_PRECED].get(token):
			return "Definor needs a precedence value."
	elif directivem.group(1) == 'rules_file:':
		rfilename = directivem.group(2).strip()
		if rfilename and rfilename.endswith('.tex') and rfilename not in mathdb[MD_RFILE]:
			mathdb[MD_RFILE].append(rfilename)
		elif rfilename and rfilename.endswith('.tex') and rfilename in mathdb[MD_RFILE]:
			pass
		elif rfilename and rfilename + '.tex' not in mathdb[MD_RFILE]:
			mathdb[MD_RFILE].append(rfilename + '.tex')
	elif directivem.group(1) == 'reductions_file:':
		redfilename = directivem.group(2).strip()
		if redfilename and redfilename.endswith('.tex') and redfilename not in mathdb[MD_REDFILE]:
			mathdb[MD_REDFILE].append(redfilename)
		elif redfilename and redfilename.endswith('.tex') and redfilename in mathdb[MD_REDFILE]:
			pass
		elif redfilename and redfilename + '.tex' not in mathdb[MD_REDFILE]:
			mathdb[MD_REDFILE].append(redfilename + '.tex')
	elif directivem.group(1) == 'notarian_formula:':
		tree = []
		tempmode = [2]
		linetailcopy = [directivem.group(2).strip()]
		addtoken(tree,'(')
#		print("linetailcopy ========", linetailcopy)
		mathparse(tempmode, linetailcopy, tree)
#		print("tree =============", tree)
		formula_definor = mathdb[MD_SYMTYPE].get(1,'dff')
		addtoken(tree, term_definor)
		if tree[-1][0][0] == -11:
			new_formula = tree[-1][1]
			print("new_formula =", new_formula)
			notarian = new_formula[1]
			print("notarian =", notarian)
			mathdb[MD_SYMTYPE][notarian] = 29 
			print( "Notarian formula: ", directivem.group(2) )
			if ';' not in new_formula:
				return "Semi-colon missing in " + str(new_formula)

#			print(tree)
#			print("new_term = = = = ", new_term)
#			input("Resulting parse")
			rd = register_definiendum(new_formula, 2)
			if type(rd) is str:
				return rd + "\n" + "Register definiendum failed on " + str(new_formula)
			promote(tree[-1],45)
			return 1	
		elif tree[-1][1][0][0] == 41: 
			pass
		elif tree[-1][1][0][0] == 15: 
			pass
		else:
#			print("tree3 ====", tree)
#			print("tree[-1]=====", tree[-1])
#			print("tree[-1][1]====", tree[-1][1])
			return "Could not parse primitive notarian formula"
			return 0
	elif directivem.group(1) == 'notarian_term:':
		tree = []
		tempmode = [2]
		linetailcopy = [directivem.group(2).strip()]
		addtoken(tree,'(')
#		print("linetailcopy ========", linetailcopy)
		mathparse(tempmode, linetailcopy, tree)
#		print("tree =============", tree)
		term_definor = 'dft'
		addtoken(tree, term_definor)
		if tree[-1][0][0] == -11:
			new_term = tree[-1][1]
			print("new_term =", new_term)
			notarian = new_term[1]
			print("notarian =", notarian)
			mathdb[MD_SYMTYPE][notarian] = 28 
			print( "Notarian term: ", directivem.group(2) )
			if ';' not in new_term:
				return "Semi-colon missing in " + str(new_term)

#			print(tree)
#			print("new_term = = = = ", new_term)
#			input("Resulting parse")
			rd = register_definiendum(new_term, 1)
			if type(rd) is str:
				return rd + "\n" + "Register definiendum failed on " + str(new_term)
			promote(tree[-1],45)
			return 1	
		elif tree[-1][1][0][0] == 40: 
			pass
		elif tree[-1][1][0][0] == 14: 
			pass
		else:
#			print("tree3 ====", tree)
#			print("tree[-1]=====", tree[-1])
#			print("tree[-1][1]====", tree[-1][1])
			return "Could not parse primitive notarian term"
			return 0
	elif directivem.group(1) == 'undefined_term:':
		tree = []
		tempmode = [2]
		linetailcopy = [directivem.group(2).strip()]
		addtoken(tree,'(')
		mathparse(tempmode, linetailcopy, tree)
		term_definor = 'dft'
		addtoken(tree, term_definor)
		if tree[-1][0][0] == -11:
			new_term = tree[-1][1]
			print("new_term =", new_term)
			print( "Primitive term: ", directivem.group(2) )
			rd = register_definiendum(new_term, 1)
			if type(rd) is str:
				return rd + "\n" + "Register definiendum failed on " + str(new_term)
			return 1	
			assert False 
			promote(tree[-1],45)
		elif tree[-1][1][0][0] == 40: 
			pass
		elif tree[-1][1][0][0] == 14: 
			pass
		else:
			return "Bad or conflicting syntax with" + str(linetailcopy)
	elif directivem.group(1) == 'undefined_formula:':
		tree = []
		tempmode = [2]
		linetailcopy = [directivem.group(2).strip()]
#		print("linetailcopy ================ ", linetailcopy)
		addtoken(tree,'(')
		mathparse(tempmode, linetailcopy, tree)
		formula_definor = mathdb[MD_SYMTYPE].get(2,'dff')
		addtoken(tree, formula_definor)
		if tree[-1][0][0] == -11:
			new_formula = tree[-1][1]
			print( "Primitive formula: ", directivem.group(2) )
			rd = register_definiendum(new_formula, 2)
			if type(rd) is str:
				return "Register definiendum failed on" + str(new_formula)
			return -1	
			promote(tree[-1],45)
		elif tree[-1][1][0][0] == 41: 
			pass
		elif tree[-1][1][0][0] == 15: 
			pass
		else:
			return "Bad or conflicting syntax with " + str(linetailcopy)
	elif directivem.group(1) == 'read_prop:':
		#Cannot do this:w: prop.readprop(directivem.group(2),mathdb)
		#Must handle this directive in check!
		pass
	elif directivem.group(1) == 'def_tracer:':
		n = directivem.group(3)
		if not n.isdigit():
				return "line number must be an integer"
		n = int(n)
		mathtf[n] = trace_fun(directivem.group(4))

	else:
		print("Warning, unrecognized directive: " + comment_line.strip())
	return 1

def mathmargin(mode,linetail):	
	newlinetail = linetail[0].lstrip()
	trimlen = len(linetail[0]) - len(newlinetail)
	if trimlen:
		blanks = linetail[0][:trimlen]
		linetail[0] = newlinetail
	if linetail[0]:
		TeXdollarm = pattern.TeXdollar.match(linetail[0])	
		if pattern.TeXcomment.match(linetail[0]):	
			linetail[0] = ''
		elif TeXdollarm: 
			linetail[0] = linetail[0][TeXdollarm.end(1):]
			mode[0] = 2
		else:
			notem = pattern.note.match(linetail[0])
			if notem:
				print("Error: Previous note unfinished.")
				mode[0] = 4
				return
			else:
				linem = pattern.line.match(linetail[0])
				if linem:
					nn = linem.start(2) - 1
					linetail[0]=linetail[0][nn:]
				elif linetail[0][:3] ==  '\\By':
					linetail[0] = ''
				else:
					mode[0] = 4  # Signal an error
	return

def fetch_tokenlist(currentline,currentpos,fetchedlist):
	userdict = mathdb[MD_MACR]
	all_fetchedlist = []

	tokenm = pattern.token.match(currentline,currentpos)
	if not tokenm:
		return currentpos 
	user_token = tokenm.group(2)
	pfch_token = userdict.get(user_token, user_token)
#
	while symtype(pfch_token) == 3:
		all_fetchedlist.append(pfch_token)
		assert pfch_token in mathdb[MD_PRECED]
		nexus = " ".join(all_fetchedlist)
		currentpos = tokenm.end(0)
		if nexus in mathdb[MD_PRECED]: 
			fetchedlist[:] = all_fetchedlist
			nextpos = currentpos
		tokenm = pattern.token.match(currentline,currentpos)
		if not tokenm:
		  return nextpos 
		user_token = tokenm.group(2)
		pfch_token = userdict.get(user_token, user_token)
	if fetchedlist == []:
		currentpos = tokenm.end(0)
		fetchedlist.append(pfch_token)
		return currentpos 
	return nextpos 
		
def mathparse(mode,linetail,tree,just_one = False):

	assert mode[0] == 2, "mode == "+ str(mode[0])
	terminateflag = False
	currentpos = 0
	currentline = linetail[0]
	lenline = len(currentline)
	tokenm = pattern.token.match(currentline,currentpos)
	if not tokenm:
#		print("currentline =======", currentline)
		print("Error: Line empty following TeX dollar sign")
		mode[0] = 4
		return		

	currentpos = tokenm.end(1)
	while currentpos < lenline: 
		endmathm = pattern.endmath.match(currentline,currentpos)
		if endmathm:
			if tree[-1][0][0] > 0:   # If the parse is done
				mode[0] = 1  # Change to text mode
			else:
				mode[0] = 3  # Change to Margin mode
			currentpos = endmathm.end(1)
			linetail[0] = currentline[currentpos:]
			return

		token_list = []
		currentpos = fetch_tokenlist(currentline,currentpos,token_list)

		if not token_list:
			print(currentline[currentpos:] + " unreadable.")
			mode[0] = 4
			return

		token = ' '.join(token_list)
		assert len(token_list) == 1 or token in mathdb[MD_PRECED]

		if tree != [] and tree[-1][0][0] > 0 : # If the parse is done
			if bracestack == []:             # If not inside braces
				if token in ['.', ',',';']:
					mode[0] = 5  # Change to end mode
					linetail[0] = currentline[currentpos:]
					return
				else:
					mode[0] = 4
					print(token + " following completed formula.")
					return
			else:                           # Still inside braces
				if just_one and bracestack[-1] != []:
					pass
#					ck = addtoken(tree, token)
				elif just_one:
					input("Got here?")
					terminateflag = True
				else:
					tree[:] = []
				if token in ['.', ',']:
					ck = 1
				else:
					ck = addtoken(tree, token)
					linetail[0] = currentline[currentpos:]
		elif tree != [] and tree[-1][0][0] < 0:
			if terminateflag:
				mode[0] = 4
				print(token + "following completed formula.")
				return
			else:
				ck = addtoken(tree, token)
		else:   #elif tree == []
			ck = addtoken(tree, token)
		
		if not ck:
			print("tree =====", tree)
			print("Parse failed at:", token)
			mode[0] = 4
			return 
 		
	linetail[0] = currentline[currentpos:]
#	print("tree ===", tree)
#	print("mode ================================= " , mode)
	return 

	
def refparse(ref):
	reflast = False
	reflist = []
	t = ref
#	if not t.strip():
#		print "Error: \\By without any justification"
#		return 0
	while t:
		t = t.lstrip()
		if not t: break
		if t[0] == '$': 
			return " dollar sign not allowed in references"
		if t[0] == '.' or t[0].isdigit():
			refmatch = pattern.ref.match(t)
			if not refmatch: 
				return t + " is not a valid reference"
				return 0
			if reflast: 
				return "Punctuator missing"
				return 0
			reflast = True
			reflist.append(refmatch.group(1)+ refmatch.group(4))
			if refmatch.group(4):
				if refmatch.group(4) not in mathdb[MD_REFD]:
					return refmatch.group(4)+ " undefined file reference key."
					return 0
			t = refmatch.group(5)
			continue
		reflast = False
		punctsmatch = pattern.puncts.match(t)
		puncts = punctsmatch.group(1)
		findsinglematch = pattern.findsingle.match(puncts)
		if findsinglematch:
			if findsinglematch.start(1) == 0:
				reflist.append(puncts[0])
				t = t[1:]
			elif findsinglematch.group(1) in reference_punctuator_list: 
				reflist.append(findsinglematch.group(1))
				t = t[findsinglematch.start(2):]
			else:
				return findsinglematch.group(1) + " invalid punctuator"
		else:
			u = punctsmatch.group(2)
			if puncts in reference_punctuator_list:
				reflist.append(puncts)
				t = u 
			else:
				return puncts + " invalid punctuator"
	return reflist

def getformula(linetail):
	mode = [2]
	parsetree = []

	fetched_tf = ''
	while linetail[0] and (mode[0]== 2 or mode[0] == 3):
		if mode[0] == 2:
			endmaths = pattern.endmath.search(linetail[0])
			if endmaths:
				stuff = linetail[0][:endmaths.start(1)]
			else:
				stuff = linetail[0].strip() 
			fetched_tf= fetched_tf + ' ' + stuff 
			mathparse(mode,linetail, parsetree)
		elif mode[0] == 3:
			mathmargin(mode,linetail)
		if not linetail[0]:
			getline(linetail)
			while linetail[0][0] == '%':
				getline(linetail) 
	if mode[0] == 4:
		return []
	if mode[0] == 5:
		print("Error: At most one term or formula allowed.")
		return []
	if parsetree[0][0][0] in [10,11]:
		return[fetched_tf,linetail[0],parsetree[0][1]]
	return [fetched_tf,linetail[0],parsetree[0]]
 
def getline(linetail,verbose=False):
# linetail = [tail of first line, index into tail, line number, list of all lines] 
	if linetail[2] == len(linetail[3]):
		linetail[0] = ''
		return linetail[0]
	linetail[0] = linetail[3][linetail[2]] 
	linetail[2] = linetail[2] + 1 # line_num
	if verbose and linetail[2] % 100 == 0:
		print(linetail[2] // 100, end=' ', flush = True)
	if linetail[0].endswith('\\extend\n'):
		linetail[0] = linetail[0][:-8]
	return linetail[0] 

def varlist(pexp):
	if type(pexp) is str:
		if symtype(pexp) in [10,11,12,13]:
			return [pexp]
		else:
			return []
	elif type(pexp) is list:
		r = []
		for t in pexp:
			s = varlist(t)
			for u in s:
				if not u in r:
					r.append(u)
		return r
	else:
		return []

#def newvarlist(list):
#	global newvarnum
# This is only called by notariancondense when some variable
# occurs in both indicial and accepted positions
#	r = []	
#	for x in list:
#		newvarnum = newvarnum + 1
#		r.append('v_{' + ("%d" % newvarnum) + '}')
#	return r

def bvarreplace(form, newbvarlist):
	if type(form) is not list: 
		return form 
	oldindvlist = indvlist(form)
	if oldindvlist:
		newindvlist = []
		for x in oldindvlist:
			newindvlist.append(newbvarlist.pop())
		nextform = indvsubst(newindvlist,oldindvlist,form)
	else:
		nextform = form[:]
	retform = [nextform[0]]
	for x in nextform[1:]:
		retform.append(bvarreplace(x, newbvarlist))
	return retform
	
def nfvlist(form):
	"""Return a list of variables with a non-free occurrence with repeats for distinct scopes"""
	retlist = []
	if type(form) is not list: 
		return []

	if type(form[0][1]) is list and len(form[0][1]) > 1:
		retlist = indvlist(form)
		for x in form[1:]:
			retlist.extend(nfvlist(x))
		return retlist

	for x in form[1:]:
		if type(x) is str: 
			pass

		elif x[0][0] == 48:
			state = 0
			for xi in x[1:]:
				if type(xi) is list: 
					assert state == 2
					retlist.extend(nfvlist(xi))
				elif symtype(xi) == 14:
					assert state == 2
					pass
				elif symtype(xi) == 10:
					if state == 0: 
						retlist.append(xi)
				elif pspeech(xi) ==  'verb':
					if state == 0:
						state = 2
				elif pspeech(xi) ==  'conjunction':
					pass
				elif pspeech(xi) ==  'reset':
					assert state == 2
					state = 0
				elif symtype(xi) == 3:
					pass
				else:
					raise RuntimeError("Scope error")
		else:
			retlist.extend(nfvlist(x))
	return retlist

def indvlist(form):
	retlist = []
	if type(form) is not list: 
		return retlist
	if len(form[0]) > 1 and type(form[0][1]) is list and len(form[0][1]) > 1:
		for x in form[0][1][1:]:
			retlist.append(form[x])
		return retlist
	for x in form[1:]:
		if type(x) is str: 
			pass
		elif x[0][0] == 48:
			state = 0
			for xi in x[1:]:
				if type(xi) is list: 
					assert state == 2
				elif symtype(xi) == 14:
					assert state == 2
				elif symtype(xi) == 10:
					if state == 0: 
						retlist.append(xi)
				elif pspeech(xi) == 'verb':
					state = 2
				elif pspeech(xi) == 'conjunction':
					pass	
				elif pspeech(xi) ==  'reset':
					assert state == 2
					state = 0
				elif symtype(xi) == 3:
					pass
#					if state == 0:
#						state = 1
				else:
					print("form = ", form)
					print("x = ", x)
					print("xi = ", xi)
					print("symtype(xi) = ", symtype(xi))
					raise RuntimeError("Scope error")
	return retlist

def indvsubst(inlist, outlist, form):
# It is expected that outlist = indvlist(form)
# and that inlist consists of fresh bound variables
	if type(form) is not list: 
		if inlist == []:
			return form
		else:
			print(form)
			raise RuntimeError("Not a bound variable form.")
	newform = [form[0]]
	if len(form[0]) == 1:
		print("form = ", form)
	if type(form[0][1]) is list and len(form[0][1]) > 1:
		definiendum = mathdb[MD_DEFS][form[1]][form[0][1][0]]
		indeflist = indvlist(definiendum)
		ibvlist = definiendum[0][1][1:]
		for n in range(1,len(form)):
			if type(definiendum[n]) is str: 
				if n in ibvlist and form[n] in outlist:
					newform.append(inlist[outlist.index(form[n])])
				else: 
					newform.append(form[n])
			elif definiendum[n][0][0] in [42,43]:
				subinlist = []
				suboutlist = []
				for m in ibvlist:
					if definiendum[m] in definiendum[n]: 
						if form[m] in outlist:
							subinlist.append(inlist[outlist.index(form[m])])
							suboutlist.append(form[m])
						else:
							raise RuntimeError("Bound variable error")
				newform.append(subst(subinlist,suboutlist,form[n]))
			else:
				newform.append(form[n])
		return newform
	for x in form[1:]:
		if type(x) is str: 
			if x in outlist:
				newform.append(inlist[outlist.index(x)])
			else:	
				newform.append(x)
		elif x[0][0] == 48:
			accum_inlist = []
			accum_outlist = []
			new_inv = ''
			new_outv = ''
			newscope = [x[0]]
			state = 0
			for xi in x[1:]:
				if type(xi) is list:
					assert  state == 2
					newscope.append(subst(accum_inlist,accum_outlist,xi))
				elif symtype(xi) == 14: 
					assert  state == 2
					newscope.append(subst(accum_inlist,accum_outlist,xi))
				elif symtype(xi) == 10: 
					if state == 0:
						if new_outv:
							accum_outlist.append(new_outv)
						if new_inv:
							accum_inlist.append(new_inv)
						if xi in outlist:
							new_outv = xi
							new_inv = inlist[outlist.index(xi)]
							newscope.append(new_inv)
						else:
							new_outv = ''
							new_inv = ''
							newscope.append(xi)
#					elif state == 1:
#						state = 2
#						newscope.append(xi)
					elif state == 2:
						newscope.append(xi)
				elif pspeech(xi) == 'verb':
					if state == 0:
						state = 2
					newscope.append(xi)
				elif pspeech(xi) == 'conjunction':
					newscope.append(xi)
				elif pspeech(xi) == 'reset':
					assert state == 2
					state = 0
					newscope.append(xi)  
				elif symtype(xi) == 3:
					newscope.append(xi)
				else:
					raise Runtime("Scope error")
			newform.append(newscope)
		else:
			newform.append(subst(inlist,outlist,x))
	return newform

def accvlist(form):
# Only called on notarian forms
	retlist = []
	if type(form) is not list: 
		return retlist
	for x in form[1:]:
		if type(x) is str: 
			pass
		elif x[0][0] == 48:
			state = 0
			newbv = ''
			accum = []
			for xi in x[1:]:
				if type(xi) is list: 
					assert state == 2
					retlist.extend(nblist(xi,accum))
				elif symtype(xi) == 14:
					assert state == 2
					pass
				elif symtype(xi) == 10:
					if state == 0:
						if newbv:
							accum.append(newbv)
						newbv = xi
					if state == 2:
						if xi not in accum:
							retlist.append(xi)
				elif pspeech(xi) == 'reset':
					assert state == 2
					state = 0
				elif pspeech(xi) == 'verb':
					state = 2
				elif pspeech(xi) == 'conjunction':
					pass
				elif symtype(xi) == 3:
					pass
				else:
					print("form = ", form)
					print("x = ", x)
					print("xi = ", xi)
					raise RuntimeError("Scope error")
	return retlist

def subst(inlist,outlist,pexp): 
	""" Indiscriminate string substitution """
	if type(pexp) is list:
		r = []
		for t in pexp:
			r.append(subst(inlist,outlist,t))
		return r
	elif type(pexp) is str:
		if pexp in outlist:
			return inlist[outlist.index(pexp)]
		else:
			return pexp
	else:  #Numbers don't get trashed
		return pexp

def nblist(form, boundlist = []):
#	print("nblist: form =======", form)
	indvars = indvlist(form)
	if type(form) is str: 
		if form in boundlist:
			return []	
		if symtype(form) in [10,11,12,13]:
			if pattern.bvar.match(form):
				return []
			return [form]
		return []	
#else	 type(form) is list: 
	assert type(form[0]) is list
	if form[0] == [11] :return form[1:]
	if form[0] == [14] or form[0] == [15]:return []
	retlist = []
	if type(form[0][1]) is list and len(form[0][1]) > 1:
		definiendum = mathdb[MD_DEFS][form[1]][form[0][1][0]]
		ibvlist = definiendum[0][1][1:]
#		print("A: ibvlist ==", ibvlist)
#		print("A: definiendum ==", definiendum)
		addvars = []
		for n in range(1,len(form)):
			if type(definiendum[n]) is str: 
				if n in ibvlist:
					pass
				elif symtype(definiendum[n]) in [10,11]:
					addvars.extend(nblist(form[n], boundlist))
			elif definiendum[n][0][0] in [42,43]:
				suboutlist = []
				for m in ibvlist:
					if definiendum[m] in definiendum[n]:
						suboutlist.append(form[m])
				addvars.extend(nblist(form[n],boundlist + suboutlist))
			else:
#				print("A: form[n] ==", form[n])
				addvars.extend(nblist(form[n],boundlist))
		for x in addvars:
			if x not in retlist and x not in boundlist:
				retlist.append(x)
#		print("A: retlist ==", retlist)
		return retlist 
	for x in form[1:]:
		if x[0][0] == 48:
			state = 0
			addvars = []
			accum = []
			newbv = ''
			for xi in x[1:]:
				if type(xi) is list: 
					assert state == 2
					addvars = addvars + nblist(xi, boundlist + accum)
				elif symtype(xi) == 14:
					assert state == 2
					addvars = addvars + nblist(xi, boundlist + accum)
				elif symtype(xi) in [10,11,12,13]:
					if state == 0:
						if newbv:
							accum.append(newbv)
						newbv = xi
#					elif state == 1:
#						state = 2
#						if xi not in boundlist:
#							if x not in accum:
#								addvars.append(xi)
					elif state == 2:
						if xi not in boundlist:
							if x not in accum:
								addvars.append(xi)
				elif pspeech(xi) == 'reset':
					state = 0
				elif pspeech(xi) == 'conjunction':
					pass
				elif pspeech(xi) == 'verb':
					state = 2
#				elif xi == ',':
#					if state == 2:
#						state = 0
				elif symtype(xi) == 3 :
					assert state == 2
					pass
#					if state == 0:
#						state = 1
		else:
			addvars = nblist(x, boundlist + indvars)
		for t in addvars:
			if t not in retlist:
				retlist.append(t)
#	print("B: retlist ==", retlist)
	return retlist


#Following function is not used.
def translate(linelist):
	""" Use the translation macros stored in the math data base
	 to translate the list of strings in linelist."""

	userdict = mathdb[MD_MACR]
	def usedict(x):
		y = x.group(1)
	#	return '\\' + userdict.get(y,y)
		return userdict.get(y,y)

	splitlist = math_mode_split(linelist)
	inmath = False
	stringlist = []
	for block in splitlist:
		if inmath: 
			newblock = [re.sub(pattern.TeX_macro,usedict, x) for x in block]
			newstring = "".join(newblock)
		else:
			newstring = "".join(block)
		stringlist.append(newstring)
		inmath = not inmath
	bigstring = "".join(stringlist)
	return bigstring.splitlines(1)
	
def pprint(exp, depth = 0):
	if type(exp) is str: 
		print(depth * '  ' + exp)
	elif len(str(exp)) + depth *2< 60: 
		print(depth * '  ' + str(exp))
	else:
		print(depth * '  ' + str(exp[0]))
		for t in exp[1:]:
			pprint(t, depth + 1) 
	return


def freshsub(vars_pexp,takenvars,fixedvars = []):	
	global newvarnum
	newvlist = []
	oldvlist = []
	for t in vars_pexp :
		if t in fixedvars:
			pass
		elif t in takenvars:
			oldvlist.append(t)
			newvar = t
			while newvar in takenvars:
				newvarnum = newvarnum + 1
				st = symtype(t)
				arity = mathdb[MD_ARITY]
				pf =  '_{' + ("%d"%newvarnum) + '}'
				if st == 10: 
					newvar = 'v' + pf
				else:
					r = arity[t]
					if st == 11:
						newvar = '\\q^{0}' + pf 
					elif st == 12:
						newvar = '\\w^{' +  ("%d"%r)  + '}' + pf
					elif st == 13:
						newvar = '\\q^{' +  ("%d"%r)  + '}' + pf 
			takenvars.append(newvar)
			newvlist.append(newvar)
		else:
			takenvars.append(t)
	return [newvlist, oldvlist]

def math_mode_split(linelist):
	""" linelist should be a list produced from a .tex file using readlines().
 The return value is a list of lists of strings.  Elements with even index 
 are text. Elements with odd index are math.  Comment strings are strings 
 whose initial symbol is a percent sign.  Every list of strings contains 
 least one string, which may be the empty string.  The length of the 
 return list is odd unless the last math mode fails to terminate."""

	if not linelist:
		return [['']]

	begin = pattern.beginmath_or_comment
	end   = pattern.endmath_or_comment


	state = 1  # reading text
	#     = 2    reading math

	blocklist = []
	block = []

	k = 0
	r = linelist[k]

	while r:
		if state == 1:
			begins = begin.search(r)
			if not begins:
				block.append(r)
				r = ''
			elif begins.group(1) == '%':
				block.append(r[:begins.start(1)])
				block.append(r[begins.start(1):])
				r = ''
			else:
				block.append(r[:begins.start(1)])
				blocklist.append(block)
				block = [begins.group(1)]
				r = r[begins.end(1):]
				state = 2
		elif state == 2:
			ends = end.search(r)
			if not ends:
				block.append(r)
				r = ''
			elif ends.group(1) == '%':
				block.append(r[:ends.start(1)])
				block.append(r[ends.start(1):])
				r = ''
			else:
				block.append(r[:ends.start(1)])
				block.append(ends.group(1))
				blocklist.append(block)
				block = []
				r = r[ends.end(1):]
				state = 1
		if r == '':
			if k + 1 < len(linelist):
				k = k + 1
				r = linelist[k]
			else:
				if block:
					blocklist.append(block)
				break
	return blocklist
							
